---
name: ai-api
description: >
  Expert in integrating AI APIs into Python applications.
  Use when implementing LLM chat completion, streaming, function/tool calling, agent loops,
  TTS/text-to-speech, voice synthesis, voice cloning, and structured outputs using
  Python urllib/requests within Kodi addon or standalone Python.
---

# AI API Integration (Python)

Expert skill for implementing AI provider APIs (LLM + TTS) in Python.

> [!IMPORTANT]
> For prompt engineering (system prompts, temperature, instructions) -> see skill `system-prompt-master`.
> For LLM provider-specific API details -> see skills:
> `openai-api-dev`, `anthropic-api-dev`, `gemini-api-dev`, `mistral-api-dev`, `groq-api-dev`, `openrouter-api-dev`
> For TTS provider-specific details -> see skills:
> `tts-voice-instructor`, `elevenlabs-tts`, `cartesia-tts`, `fish-audio-tts`, `playht-tts`

## Architecture (Kodi Addon)

```
Kodi Python Addon          AI Provider
─────────────────    ─────────────────
urllib.request    ->  OpenAI / Gemini / Mistral / etc.
JSON response     <-  Chat completion / TTS audio
```

### Security Rules
- API keys in Kodi addon settings or environment variables -- never hardcoded
- Validate and sanitize all LLM output before displaying in UI
- All network calls in background threads (Kodi GUI is not thread-safe)

---

## Chat Completion (Python)

```python
import json
import urllib.request

def call_llm(messages, model='gpt-4.1-mini', temperature=0.7, api_key=None):
    """Call LLM API and return response text."""
    payload = json.dumps({
        'model': model,
        'messages': messages,
        'temperature': temperature
    }).encode('utf-8')

    req = urllib.request.Request(
        'https://api.openai.com/v1/chat/completions',
        data=payload,
        headers={
            'Authorization': f'Bearer {api_key}',
            'Content-Type': 'application/json'
        }
    )

    with urllib.request.urlopen(req) as resp:
        data = json.loads(resp.read().decode('utf-8'))
        return data['choices'][0]['message']['content']
```

---

## Streaming (Python)

```python
def stream_llm(messages, model, api_key, on_chunk):
    """Stream LLM response token by token."""
    payload = json.dumps({
        'model': model,
        'messages': messages,
        'stream': True
    }).encode('utf-8')

    req = urllib.request.Request(
        'https://api.openai.com/v1/chat/completions',
        data=payload,
        headers={
            'Authorization': f'Bearer {api_key}',
            'Content-Type': 'application/json'
        }
    )

    with urllib.request.urlopen(req) as resp:
        for line in resp:
            line = line.decode('utf-8').strip()
            if not line.startswith('data: ') or line == 'data: [DONE]':
                continue
            chunk = json.loads(line[6:])
            content = chunk.get('choices', [{}])[0].get('delta', {}).get('content')
            if content:
                on_chunk(content)
```

---

## Tools / Function Calling

### Tool Definition
```python
tools = [{
    'type': 'function',
    'function': {
        'name': 'get_question',
        'description': 'Get a quiz question for the game',
        'parameters': {
            'type': 'object',
            'properties': {
                'difficulty': {'type': 'string', 'enum': ['easy', 'medium', 'hard']},
                'category': {'type': 'string', 'description': 'Question category'}
            },
            'required': ['difficulty']
        }
    }
}]
```

### Agent Loop
```python
def agent_loop(messages, tools, max_iterations=10):
    """Run agent loop with tool calling."""
    for i in range(max_iterations):
        response = call_llm_with_tools(messages, tools)
        choice = response['choices'][0]

        if choice['finish_reason'] != 'tool_calls':
            return choice['message']['content']

        messages.append(choice['message'])
        for call in choice['message']['tool_calls']:
            fn_name = call['function']['name']
            args = json.loads(call['function']['arguments'])
            result = execute_function(fn_name, args)
            messages.append({
                'role': 'tool',
                'tool_call_id': call['id'],
                'content': json.dumps(result)
            })
    return 'Max iterations reached'
```

---

## Structured Output (JSON Schema)

```python
payload['response_format'] = {
    'type': 'json_schema',
    'json_schema': {
        'name': 'quiz_question',
        'strict': True,
        'schema': {
            'type': 'object',
            'properties': {
                'question': {'type': 'string'},
                'answers': {'type': 'array', 'items': {'type': 'string'}},
                'correct_index': {'type': 'integer'}
            },
            'required': ['question', 'answers', 'correct_index']
        }
    }
}
```

---

## TTS (Text-to-Speech)

```python
def generate_tts(text, voice='alloy', api_key=None, output_path=None):
    """Generate TTS audio and save to file."""
    payload = json.dumps({
        'model': 'gpt-4o-mini-tts',
        'input': text,
        'voice': voice
    }).encode('utf-8')

    req = urllib.request.Request(
        'https://api.openai.com/v1/audio/speech',
        data=payload,
        headers={
            'Authorization': f'Bearer {api_key}',
            'Content-Type': 'application/json'
        }
    )

    with urllib.request.urlopen(req) as resp:
        audio_data = resp.read()
        if output_path:
            with open(output_path, 'wb') as f:
                f.write(audio_data)
        return audio_data
```

> For voice instruction engineering -> see skill `tts-voice-instructor`.

---

## Provider Compatibility

| Feature | OpenAI | Anthropic | Gemini | Mistral | Groq | OpenRouter |
|---------|--------|-----------|--------|---------|------|------------|
| Chat Completion | yes | yes (Messages API) | yes | yes | yes | yes |
| Streaming | yes SSE | yes SSE (own events) | yes SSE | yes SSE | yes SSE | yes SSE |
| Tools/Function Calling | yes | yes (tool_use blocks) | yes | yes | yes | yes |
| JSON Schema Output | yes strict | no | yes | yes | no | depends on model |
| TTS | yes | no | no | no | yes (Whisper) | no |
| Vision | yes | yes | yes | yes (Pixtral) | no | depends on model |
| OpenAI-compatible | yes | no | no | yes | yes | yes |

> [!WARNING]
> **Anthropic** uses its own Messages API format -- NOT OpenAI-compatible.
> See `anthropic-api-dev` skill for format differences.

## Error Handling

```python
import time

def call_llm_with_retry(payload, max_retries=3):
    """Call LLM with exponential backoff retry."""
    for i in range(max_retries):
        try:
            return call_llm(payload)
        except Exception as e:
            xbmc.log(f"[Milionar] LLM API error (attempt {i+1}): {e}", xbmc.LOGERROR)
            if i < max_retries - 1:
                time.sleep(2 ** i * 0.1)
    raise RuntimeError('LLM API failed after retries')
```

## Kodi-Specific Rules
- All API calls MUST run in background threads (never block UI)
- Use `xbmc.Monitor().waitForAbort()` for cancellation
- Store API keys in addon settings via `xbmcaddon.Addon().getSetting()`
- Cache responses when appropriate (save API costs)
- Log API errors with `xbmc.log()` at `LOGERROR` level

## Checklist
- [ ] API keys in addon settings, never hardcoded
- [ ] Network calls in background threads
- [ ] Agent loop has max iteration limit
- [ ] Tool definitions use proper JSON Schema
- [ ] Error handling with retry + backoff
- [ ] LLM output sanitized before display

## Related Skills
- `system-prompt-master` -- prompt engineering
- `kodi-logic-core` -- Kodi Python API, threading, lifecycle
