---
trigger: always_on
description: Always active -- modular Python code architecture for Kodi addon
---

# Modular Code Architecture

> [!CAUTION]
> **NEVER CREATE MONOLITHIC FILES!**
> Large files prevent efficient LLM agent work due to limited context window.

## Why Modularity

1. **LLM context** -- agent loads only needed modules, not thousands of lines
2. **Maintenance** -- changing one function = editing one file
3. **Testing** -- isolated units are easier to test
4. **Parallel work** -- multiple changes without conflicts

---

## Principles

### Single Responsibility
- **One function = one purpose**
- **One file = one logical unit**
- If you describe a file using "and" or "also", split it

### Explicit Dependencies
- All imports at the top of the file
- No cross-module global variables
- Dependency injection over hardcoded dependencies

### Naming
- File name = main purpose (`ai_chat.py` -> AI chat logic)
- Folders = logical groups (`/providers`, `/utils`, `/services`)

---

## Kodi Python Addon Structure

```
/addon_root
  main.py                  # Entry point (router only)
  service.py               # Background service
  /wcs                     # Main code package
    __init__.py
    /providers             # Content providers (one per source)
    /services              # Business logic modules
    /utils                 # Utility functions
    /gui                   # WindowXML classes and dialogs
    /api                   # External API integrations
```

### Rules
- **Entry point** (`main.py`) -- only routing, no business logic
- **One class per file** (except small related helpers)
- **Standard library + Kodi API only** -- no pip, no external packages
- **Relative imports** within package, Kodi special paths for resources

### Correct Structure Example
```python
# BAD: one_big_file.py (2000 lines)
def search_tmdb(): ...
def play_video(): ...
def show_dialog(): ...
# ... 80 more functions

# GOOD: modular structure
# wcs/providers/tmdb.py
from ..utils.http import fetch_json
def search(query): ...

# wcs/utils/http.py
def fetch_json(url, params=None): ...
```

---

## When Larger Files Are OK

Exceptions where more code in one file is acceptable:

1. **Data files** -- constants, configuration, definitions
2. **Single complex algorithm** -- one complex function with helper methods
3. **Legacy code** -- during gradual refactoring (mark with comment)

> [!IMPORTANT]
> Even in these cases, prefer splitting if it makes sense.

---

## Refactoring Checklist

When creating new code or modifying existing:

- [ ] Is each file under ~200 lines? (flexible, depends on context)
- [ ] Does each file have one clear purpose?
- [ ] Are all dependencies explicit (imports)?
- [ ] Can the module be understood independently?
- [ ] Is naming clear and consistent?

---

## Anti-patterns

### God Object
One file/class does everything.

### Copy-Paste
Duplicated code instead of shared function.

### Deep Nesting
10 levels of nested folders -- max 3-4 levels.

### Circular Dependencies
Module A imports B, B imports A.

### Mixed Concerns
UI logic mixed with business logic and API calls in one file.
