---
name: web-performance-optimization
description: >
  Use when optimizing website and web application performance including loading speed,
  Core Web Vitals, Lighthouse scores, caching strategies, and runtime performance.
---

# Web Performance Optimization

You are a web performance expert who optimizes loading speed, Core Web Vitals, caching, and runtime performance for vanilla HTML/CSS/JS/PHP stacks without build tools.

## When to use

- Slow page loading
- Core Web Vitals optimization (LCP, FID/INP, CLS)
- Image and asset optimization
- Caching strategies
- Debugging performance issues

## Procedure

### 1. Measure current state
- Lighthouse audit
- Core Web Vitals (LCP, INP, CLS)
- Network waterfall in DevTools
- Identify bottlenecks

### 2. Identify problems
- Unoptimized images
- Blocking CSS/JS resources
- Slow server response (PHP)
- Missing cache headers
- Layout shifts
- Long JS tasks on main thread

### 3. Prioritize
High-impact first:
1. Critical rendering path
2. Images
3. CSS/JS optimization
4. Server response (PHP)
5. Caching

### 4. Implement

### 5. Verify
- Compare before/after metrics
- Test on mobile + slow 3G

---

## Image optimization

```html
<!-- Modern formats + responsive -->
<picture>
  <source srcset="/img/hero.avif" type="image/avif">
  <source srcset="/img/hero.webp" type="image/webp">
  <img 
    src="/img/hero.jpg" 
    alt="Hero"
    width="1200" 
    height="600"
    loading="eager"
    fetchpriority="high"
  >
</picture>

<!-- Lazy loading for below-the-fold -->
<img src="/img/product.jpg" alt="Product" width="400" height="300" loading="lazy">
```

**Rules:**
- ALWAYS `width` + `height` (CLS prevention)
- Above-the-fold: `loading="eager"` + `fetchpriority="high"`
- Below-the-fold: `loading="lazy"`
- Formats: AVIF > WebP > JPEG
- Max size: ~200KB per image
- Preload hero: `<link rel="preload" as="image" href="/img/hero.avif">`

---

## CSS optimization

```html
<!-- Critical CSS inline -->
<style>
  /* Just minimum for above-the-fold rendering */
  :root { --bg: #0a0a0a; --text: #fff; }
  body { background: var(--bg); color: var(--text); margin: 0; }
</style>

<!-- Rest CSS async -->
<link rel="preload" href="/css/styles.css" as="style" onload="this.onload=null;this.rel='stylesheet'">
<noscript><link rel="stylesheet" href="/css/styles.css"></noscript>
```

**Rules:**
- Critical CSS inline in `<head>` (max 14KB)
- Remove unused CSS
- `@media print` in separate file with `media="print"`
- CSS containment: `contain: layout style paint`

---

## JavaScript optimization

```html
<!-- Defer non-critical scripts -->
<script src="/js/main.js" defer></script>

<!-- Lazy load heavy modules -->
<script>
  // Load analytics after page interactive
  window.addEventListener('load', () => {
    const s = document.createElement('script');
    s.src = '/js/analytics.js';
    s.async = true;
    document.body.appendChild(s);
  });
</script>
```

**Rules:**
- `defer` for all scripts (not `async` unless order doesn't matter)
- Dynamic import for heavy modules
- Split long tasks (>50ms) into smaller chunks
- `requestIdleCallback()` for non-critical work
- Event delegation instead of thousands of listeners

---

## PHP / Server optimization

```php
// Cache headers for static assets
header('Cache-Control: public, max-age=31536000, immutable');

// ETag for dynamic content
$etag = md5($content);
header("ETag: \"$etag\"");
if (isset($_SERVER['HTTP_IF_NONE_MATCH']) && trim($_SERVER['HTTP_IF_NONE_MATCH']) === "\"$etag\"") {
    http_response_code(304);
    exit;
}

// Gzip compression
if (strpos($_SERVER['HTTP_ACCEPT_ENCODING'] ?? '', 'gzip') !== false) {
    ob_start('ob_gzhandler');
}
```

**Rules:**
- Static assets: `Cache-Control: max-age=31536000, immutable` + versioning (`?v=`)
- API responses: `Cache-Control: private, max-age=60`
- Gzip/Brotli compression
- Minimize DB queries
- Connection pooling

---

## CLS (Cumulative Layout Shift)

```css
/* Reserve space for dynamic content */
img, video { aspect-ratio: 16/9; width: 100%; height: auto; }

/* Skeleton loader */
.skeleton {
  min-height: 200px;
  background: linear-gradient(90deg, #1a1a1a 25%, #2a2a2a 50%, #1a1a1a 75%);
  background-size: 200% 100%;
  animation: shimmer 1.5s infinite;
}
@keyframes shimmer {
  0% { background-position: 200% 0; }
  100% { background-position: -200% 0; }
}
```

**CLS prevention:**
- ALWAYS `width`/`height` on `<img>` and `<video>`
- Fonts: `font-display: swap` + `<link rel="preload">`
- Never insert content above existing (push down)
- Skeleton loaders for async content

---

## Core Web Vitals targets

| Metric | Good | Poor |
|--------|------|------|
| **LCP** | < 2.5s | > 4.0s |
| **INP** | < 200ms | > 500ms |
| **CLS** | < 0.1 | > 0.25 |
| **TTFB** | < 600ms | > 1.8s |

---

## Checklist

### Images
- [ ] Modern formats (WebP, AVIF)
- [ ] `width`/`height` on all `<img>`
- [ ] `loading="lazy"` below-the-fold
- [ ] Max ~200KB per image

### CSS/JS
- [ ] Critical CSS inline
- [ ] `defer` on script tags
- [ ] No render-blocking resources
- [ ] Unused CSS/JS removed

### Server
- [ ] Cache headers properly configured
- [ ] Gzip/Brotli compression
- [ ] TTFB < 600ms

### Core Web Vitals
- [ ] LCP < 2.5s
- [ ] INP < 200ms
- [ ] CLS < 0.1
