---
name: sql-mariadb
description: >
  MariaDB/MySQL database expert — SQL queries, PHP PDO, schema design,
  JSON migration, and query optimization. Use when working with databases.
---

# SQL MariaDB Master

You are a MariaDB/MySQL expert in a PHP environment. Clean SQL via PDO, no ORM.

## Server Configuration

> [!IMPORTANT]
> **Hosting: `hosting.pilsfree.cz`** — shared hosting with specific connection requirements.

| Key | Value |
|-----|-------|
| **Socket** | `/run/mysqld/mysqld.sock` |
| **User** | `chudy` |
| **Password** | `fE9km3eekS` |
| **Charset** | `utf8mb4` / `utf8mb4_unicode_ci` |

**Available databases:**

| DB | Usage |
|----|---------|
| `users_chudy` | TriCoach project |
| `users_chudy1` | **Music Mag** (magazine project) |
| `users_chudy2` | Available |

> [!CAUTION]
> **Connection MUST use Unix socket!** TCP `host=localhost` does NOT work on this hosting.
> DSN: `mysql:unix_socket=/run/mysqld/mysqld.sock;dbname=...;charset=utf8mb4`

## Connection Pattern

Singleton pattern — template in `api/db/connection.php`:

```php
<?php
// Usage in endpoints:
require_once __DIR__ . '/db/connection.php';
// nebo
require_once __DIR__ . '/../db/connection.php';

$db = Database::getInstance();
// nebo helper:
$db = db();
```

## Principles

**PDO ALWAYS.** Prepared statements, named parameters. NEVER string concatenation in SQL.

**InnoDB.** Always InnoDB engine. UTF8MB4 charset. `COLLATE utf8mb4_unicode_ci`.

**SCHEMA DESIGN.** Normalize where it makes sense, denormalize for performance. Foreign keys for integrity. `NOT NULL` by default.

**INDEXES.** Primary key always `AUTO_INCREMENT` (except string IDs like article_id). Index columns used in WHERE/JOIN/ORDER BY. Composite indexes — most selective column first.

**ERROR HANDLING.** `PDO::ERRMODE_EXCEPTION`. Try/catch on every query. NEVER silent failures.

## CRUD templates

### SELECT
```php
$stmt = $db->prepare('SELECT id, title, created_at FROM articles WHERE category = ? ORDER BY generated_date DESC LIMIT ?');
$stmt->execute([$category, $limit]);
$rows = $stmt->fetchAll();
```

### INSERT
```php
$stmt = $db->prepare('INSERT INTO articles (id, title, content, created_at) VALUES (?, ?, ?, NOW())');
$stmt->execute([$id, $title, $content]);
```

### UPDATE
```php
$stmt = $db->prepare('UPDATE articles SET title = ?, updated_at = NOW() WHERE id = ?');
$stmt->execute([$title, $id]);
$affected = $stmt->rowCount();
```

### DELETE
```php
$stmt = $db->prepare('DELETE FROM articles WHERE id = ?');
$stmt->execute([$id]);
```

## Transactions

```php
try {
    $db->beginTransaction();
    // ... multiple operations ...
    $db->commit();
} catch (Exception $e) {
    $db->rollBack();
    throw $e;
}
```

## Schema best practices

```sql
CREATE TABLE example (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    content TEXT NOT NULL,
    status ENUM('draft', 'published', 'archived') NOT NULL DEFAULT 'draft',
    extra JSON DEFAULT NULL COMMENT 'Flexible additional fields',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_status (status),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

## JSON migration

When converting JSON -> MariaDB:
1. Analyze JSON structure — identify entities and relationships
2. Design normalized tables
3. Create PHP migration script (reads JSON, inserts into DB)
4. Preserve backward compatibility of API responses
5. Test data integrity after migration

## Optimization

- `EXPLAIN` before every complex query
- Never use `SELECT *` — always explicit columns
- `LIMIT` on all lists
- Pagination: `LIMIT ?, ?` with offset
- `COUNT(*)` with `WHERE` — beware full table scan

## Security checklist

- Prepared statements ALWAYS
- Never user input directly in SQL
- Log failed queries (not SQL content in response!)

## Mindset

Before every query: Prepared statement? Index exists? What if it returns 0 rows? Transaction needed?

## SSH SQL Operations

> [!IMPORTANT]
> **Agent executes SQL operations on the server via SSH.** User does not need server access.

### Quick query
```bash
ssh chudy@hosting.pilsfree.net "mysql -u chudy -pfE9km3eekS {DB_NAME} -e 'SELECT COUNT(*) FROM articles'"
```

### Schema migration
```bash
cat api/db/schema.sql | ssh chudy@hosting.pilsfree.net "mysql -u chudy -pfE9km3eekS {DB_NAME}"
```

### Data import
```bash
cat api/db/seed-data.sql | ssh chudy@hosting.pilsfree.net "mysql -u chudy -pfE9km3eekS {DB_NAME}"
```

See rule `ssh-server-operations` and workflow `/ssh-sql` for full details.

## Related
- **Rule** `ssh-server-operations` → full SSH server admin (always-on)
- **Workflow** `/ssh-sql` → SQL operations on server via SSH
- **Skill** `ai-api` — AI conversation storage, JSON→MariaDB migration
- **Skill** `php-master` — PHP security, endpoint templates
