---
name: playht-tts
description: >
  Use this skill when integrating PlayHT TTS API for text-to-speech,
  voice cloning, conversational dialogue synthesis, or streaming audio.
  Covers REST/WebSocket API via PHP proxy, models, and voices.
---

# PlayHT TTS Skill

You are an expert at integrating PlayHT TTS API into web applications using PHP proxy and vanilla JavaScript.

## Overview

PlayHT provides ultra-realistic text-to-speech with conversational AI:
- **900+ voices** — 140+ languages, various styles
- **Play3.0** — multilingual streaming model
- **PlayDialog** — conversational dialogue, context-aware prosody
- **Voice cloning** — custom AI voice replicas
- **SSML support** — pauzy, tempo, intonace
- **WebSocket + HTTP streaming** — real-time audio

## Current Models

| Model | Use Case | Languages | Notes |
|-------|----------|-----------|-------|
| `Play3.0` | General TTS, streaming | Multilingual | Default streaming model |
| `Play3.0-mini` | Lowest latency | Multilingual | Fastest, up to 20K chars |
| `PlayDialog` | Conversational | EN + multi | Context-aware prosody, emotion |
| `PlayDialog-turbo` | Fast dialogue | EN | Powered by Groq silicon |

> [!IMPORTANT]
> `Play3.0-mini` for lowest latency.
> `PlayDialog` for most natural conversational speech.

## API Base

```
https://api.play.ht/api/v2/
```

Authentication:
- `Authorization: Bearer $PLAYHT_API_KEY`
- `X-USER-ID: $PLAYHT_USER_ID`

## Key Endpoints

| Endpoint | Method | Purpose |
|----------|--------|---------|
| `/api/v2/tts/stream` | POST | HTTP streaming TTS |
| `wss://api.play.ht/...` | WS | WebSocket real-time TTS |
| `/api/v2/voices` | GET | List prebuilt voices |
| `/api/v2/batch` | POST | Batch TTS job |

## Quick Start (PHP)

```php
<?php
$payload = [
    'text' => $text,
    'voice' => 'en-US-MichaelNeural',
    'output_format' => 'mp3',
    'voice_engine' => 'Play3.0-mini',
    'sample_rate' => 24000
];

$ch = curl_init('https://api.play.ht/api/v2/tts/stream');
curl_setopt_array($ch, [
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => [
        'Authorization: Bearer ' . getenv('PLAYHT_API_KEY'),
        'X-USER-ID: ' . getenv('PLAYHT_USER_ID'),
        'Content-Type: application/json',
        'Accept: audio/mpeg'
    ],
    CURLOPT_POSTFIELDS => json_encode($payload),
    CURLOPT_RETURNTRANSFER => true
]);
$audioData = curl_exec($ch);
curl_close($ch);

header('Content-Type: audio/mpeg');
echo $audioData;
```

## Streaming (PHP proxy)

```php
<?php
header('Content-Type: audio/mpeg');
header('Transfer-Encoding: chunked');

$ch = curl_init('https://api.play.ht/api/v2/tts/stream');
curl_setopt_array($ch, [
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => [
        'Authorization: Bearer ' . getenv('PLAYHT_API_KEY'),
        'X-USER-ID: ' . getenv('PLAYHT_USER_ID'),
        'Content-Type: application/json'
    ],
    CURLOPT_POSTFIELDS => json_encode($payload),
    CURLOPT_WRITEFUNCTION => function($ch, $data) {
        echo $data;
        ob_flush();
        flush();
        return strlen($data);
    }
]);
curl_exec($ch);
```

## Output Formats

- `mp3`
- `wav`
- `mulaw`
- `flac`
- `ogg`
- Sample rates: 8K, 16K, 24K, 44.1K, 48K Hz

## API Docs

- [Quickstart](https://docs.play.ht/reference/api-getting-started)
- [Authentication](https://docs.play.ht/reference/authentication)
- [Models](https://docs.play.ht/reference/models)
- [HTTP Streaming](https://docs.play.ht/reference/api-generate-tts-audio-stream)
- [WebSocket API](https://docs.play.ht/reference/playht-tts-websocket-api)
- [Prebuilt Voices](https://docs.play.ht/reference/list-of-prebuilt-voices)
- [Latency Tips](https://docs.play.ht/reference/techniques-to-guarantee-the-lowest-latency-1)
- [Dialog Turbo (Groq)](https://docs.play.ht/reference/groq)

## Related Skills
- `ai-api` — AI integration patterns (LLM + TTS), PHP proxy architecture
- `tts-voice-instructor` — voice instruction engineering (OpenAI TTS)
