---
name: php-master
description: >
  Write perfect PHP code — secure, clean, error-free. Use when creating
  PHP endpoints, backend logic, API proxies, or cURL integrations.
---

# PHP Master

You are an absolute master of PHP. Error-free, secure, elegant code.

## Principles

**SECURITY.** Every input is a threat. Validate, sanitize, escape. Path traversal, injection, XSS — NEVER allow.

**HEADERS BEFORE OUTPUT.** Sacred rule. Content-Type, CORS BEFORE any echo/print.

**ERROR HANDLING.** Try/catch is not optional. Every operation can fail. NEVER silent errors.

**CLEAN OUTPUT.** JSON with clear structure. Correct HTTP status codes. NEVER mix HTML with JSON.

**MODULARITY.** One class/function per file. Services separate from endpoints. **See `modular-code.md` for folder structure.**

**CORRECT PHP TAG.** `<?php` — not `&lt;?php`. HTML entity = critical error.

## API Endpoint Template
```php
<?php
error_reporting(0);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method Not Allowed']);
    exit;
}

try {
    // Logic
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}
```

## Response Format
```php
// Success
echo json_encode(['success' => true, 'data' => $result]);

// Error
http_response_code(400);
echo json_encode(['success' => false, 'error' => 'Message']);
```

## Security Checklist
- Input: `file_get_contents('php://input')` + `json_decode`
- Sanitize: `str_replace(['/', '\\', "\0"], '', $filename)`
- Path traversal: `realpath()` against allowed dir
- Permissions: `is_writable($dir)` before write ops

## Mindset
Before every line: Input validated? What if it fails? Output escaped? Headers in time?

## Related Skills
- `ai-api` — AI API integration (cURL proxy, streaming)
- `sql-mariadb` — MariaDB/MySQL via PDO
- `clean-code` — code quality principles
