---
name: openrouter-api-dev
description: >
  Use this skill when building applications with OpenRouter unified LLM proxy,
  accessing 500+ models from 60+ providers through a single API.
  OpenAI-compatible, supports streaming, tool calling, and model routing.
---

# OpenRouter API Development Skill

You are an expert at integrating OpenRouter unified LLM API into web applications using PHP proxy and vanilla JavaScript.

## Overview

OpenRouter provides a single API gateway to 500+ models from 60+ providers:
- **Unified API** — one endpoint, one key, all providers
- **OpenAI-compatible** — drop-in replacement for OpenAI endpoints
- **Streaming** — real-time SSE output
- **Tool Calling** — standardized across providers
- **Model Routing** — automatic fallbacks and cost optimization
- **Provider Preferences** — choose specific providers per request

## Why Use OpenRouter

- **Single integration** for all providers (OpenAI, Anthropic, Google, Mistral, Meta, etc.)
- **Automatic fallbacks** — if one provider fails, routes to another
- **Cost optimization** — choose cheapest provider for each model
- **No vendor lock-in** — switch models without code changes
- **Free tier** — some models available for free

## API Base

```
https://openrouter.ai/api/v1/
```

Authentication: `Authorization: Bearer $OPENROUTER_API_KEY`

Optional headers:
- `HTTP-Referer: https://your-site.com` (for rankings)
- `X-Title: Your App Name` (for dashboard)

## Key Endpoints

| Endpoint | Purpose |
|----------|---------|
| `/api/v1/chat/completions` | Chat completion (OpenAI format) |
| `/api/v1/models` | List available models |
| `/api/v1/auth/key` | Check API key credits |
| `/api/v1/generation` | Get generation details |

## Model Naming

Models use `provider/model-name` format:
```
openai/gpt-4.1
anthropic/claude-sonnet-4.5
google/gemini-3-flash-preview
mistralai/mistral-large-2512
meta-llama/llama-3.3-70b-instruct
```

## Quick Start (PHP)

```php
<?php
$payload = [
    'model' => 'openai/gpt-4.1-mini',  // provider/model format
    'messages' => [
        ['role' => 'system', 'content' => 'You are helpful.'],
        ['role' => 'user', 'content' => $userMessage]
    ]
];

$ch = curl_init('https://openrouter.ai/api/v1/chat/completions');
curl_setopt_array($ch, [
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => [
        'Authorization: Bearer ' . getenv('OPENROUTER_API_KEY'),
        'Content-Type: application/json',
        'HTTP-Referer: https://your-site.com'
    ],
    CURLOPT_POSTFIELDS => json_encode($payload),
    CURLOPT_RETURNTRANSFER => true
]);
$response = json_decode(curl_exec($ch), true);
curl_close($ch);
```

## Tool Calling

Same as OpenAI format — standardized across providers:
```php
$payload['tools'] = [[
    'type' => 'function',
    'function' => [
        'name' => 'get_weather',
        'parameters' => [
            'type' => 'object',
            'properties' => ['city' => ['type' => 'string']],
            'required' => ['city']
        ]
    ]
]];
```

> [!TIP]
> Not all models support tool calling. Check model capabilities via `/api/v1/models`.

## Provider Routing

```php
$payload['provider'] = [
    'order' => ['Anthropic', 'OpenAI'],  // prefer Anthropic, fallback OpenAI
    'allow_fallbacks' => true
];
```

## Cost Control

```php
// Use auto-routing for cheapest option
$payload['model'] = 'openrouter/auto';  // auto-selects best model

// Or sort by price
$payload['provider'] = [
    'sort' => 'price'  // cheapest first
];
```

## API Docs (source of truth)

**llms.txt URL**: `https://openrouter.ai/docs/llms.txt`

Key documentation pages:

- [Quick Start](https://openrouter.ai/docs/quickstart)
- [Models](https://openrouter.ai/docs/models)
- [Tool Calling](https://openrouter.ai/docs/tool-calling)
- [Streaming](https://openrouter.ai/docs/streaming)
- [Provider Routing](https://openrouter.ai/docs/provider-routing)
- [API Keys](https://openrouter.ai/docs/api-keys)

## Related Skills
- `ai-api` — general AI integration patterns (streaming, agent loops, PHP proxy)
- `system-prompt-master` — prompt engineering
