---
name: openai-api-dev
description: >
  Use this skill when building applications with OpenAI models,
  OpenAI API, implementing chat completions, streaming, function/tool calling,
  structured outputs, TTS, image generation, or needing current model specs.
  Covers REST API usage via PHP cURL proxy, model selection, and capabilities.
---

# OpenAI API Development Skill

You are an expert at integrating OpenAI API into web applications using PHP proxy and vanilla JavaScript.

## Overview

The OpenAI API provides access to GPT models and specialized AI capabilities:
- **Chat Completion** — text generation, conversation, reasoning
- **Streaming** — real-time token-by-token output via SSE
- **Function/Tool Calling** — let models invoke your functions
- **Structured Output** — generate valid JSON matching your schema
- **TTS** — text-to-speech with voice instructions
- **Image Generation** — GPT Image models
- **Embeddings** — text embeddings for semantic search
- **Realtime API** — audio/speech in real-time

## Current OpenAI Models

- `gpt-5.3-codex`: latest coding model, 25% faster, self-improving
- `gpt-5.2-codex`: agentic coding, multi-file work, defensive cybersecurity
- `gpt-5.2`: top agentic model, advanced reasoning ($1.75/$14.00 per 1M tokens)
- `gpt-5.1`: balanced reasoning model ($1.25/$10.00 per 1M tokens)
- `gpt-4.1`: smartest non-reasoning model, 1M context
- `gpt-4.1-mini`: balanced performance/cost
- `gpt-4.1-nano`: fastest, most cost-effective
- `o3`: specialized reasoning & agentic tasks ($2.00/$8.00)
- `o4-mini`: fast reasoning model ($2.00/$8.00)
- `gpt-image-1.5`: image generation & editing
- `gpt-4o-mini-tts`: TTS with voice instructions

> [!CAUTION]
> **GPT-4o is fully deprecated** (February 2026). Do NOT use `gpt-4o` or `gpt-4o-latest`.
> Migrate to `gpt-4.1-mini` (drop-in replacement) or `gpt-5.2` for advanced tasks.
> **Assistants API** sunsets August 2026 — use Responses API instead.

## Reasoning Parameters

### `reasoning_effort` (o3, o4-mini, gpt-5.x)
Controls how much reasoning the model performs before responding:
- `low` — fast, minimal reasoning
- `medium` — balanced
- `high` — maximum reasoning depth (default for o-series)

```php
$payload['reasoning_effort'] = 'medium';
```

### `max_completion_tokens` (required for newer models)
GPT-5.x, o3, o4-mini require `max_completion_tokens` instead of `max_tokens`:

```php
// ❌ Breaks with newer models
$payload['max_tokens'] = 10000;

// ✅ Correct for gpt-5.x, o3, o4-mini
$payload['max_completion_tokens'] = 10000;
```

## API Base

```
https://api.openai.com/v1/
```

Authentication: `Authorization: Bearer $OPENAI_API_KEY`

## Key Endpoints

| Endpoint | Purpose |
|----------|---------|
| `/v1/chat/completions` | Chat completion (main) |
| `/v1/responses` | Responses API (newer) |
| `/v1/audio/speech` | TTS |
| `/v1/audio/transcriptions` | Speech-to-text |
| `/v1/images/generations` | Image generation |
| `/v1/embeddings` | Text embeddings |

## Quick Start (PHP)

```php
<?php
$payload = [
    'model' => 'gpt-4.1-mini',
    'messages' => [
        ['role' => 'system', 'content' => 'You are helpful.'],
        ['role' => 'user', 'content' => $userMessage]
    ],
    'temperature' => 0.7
];

$ch = curl_init('https://api.openai.com/v1/chat/completions');
curl_setopt_array($ch, [
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => [
        'Authorization: Bearer ' . getenv('OPENAI_API_KEY'),
        'Content-Type: application/json'
    ],
    CURLOPT_POSTFIELDS => json_encode($payload),
    CURLOPT_RETURNTRANSFER => true
]);
$response = json_decode(curl_exec($ch), true);
curl_close($ch);
```

## Tool Calling

```php
$payload['tools'] = [[
    'type' => 'function',
    'function' => [
        'name' => 'get_weather',
        'description' => 'Get weather for a city',
        'parameters' => [
            'type' => 'object',
            'properties' => [
                'city' => ['type' => 'string']
            ],
            'required' => ['city']
        ]
    ]
]];
// Response: choices[0].message.tool_calls[].function.{name, arguments}
// Follow up: role: "tool", tool_call_id, content
```

## Structured Output

```php
$payload['response_format'] = [
    'type' => 'json_schema',
    'json_schema' => [
        'name' => 'result',
        'strict' => true,
        'schema' => [
            'type' => 'object',
            'properties' => [...],
            'required' => [...]
        ]
    ]
];
```

## API Docs (source of truth)

**llms.txt URL**: `https://platform.openai.com/docs/llms.txt`

Each doc page available at `/api/docs/<slug>.md`. Key pages:

- [Models](https://platform.openai.com/docs/models)
- [Chat Completions](https://platform.openai.com/docs/guides/chat-completions)
- [Function Calling](https://platform.openai.com/docs/guides/function-calling)
- [Structured Outputs](https://platform.openai.com/docs/guides/structured-outputs)
- [Streaming](https://platform.openai.com/docs/guides/streaming)
- [Text-to-Speech](https://platform.openai.com/docs/guides/text-to-speech)
- [Image Generation](https://platform.openai.com/docs/guides/image-generation)
- [Responses API](https://platform.openai.com/docs/guides/responses-vs-chat-completions)

## Related Skills
- `ai-api` — general AI integration patterns (streaming, agent loops, PHP proxy)
- `tts-voice-instructor` — TTS voice instructions for `gpt-4o-mini-tts`
- `system-prompt-master` — prompt engineering
