---
name: mistral-api-dev
description: >
  Use this skill when building applications with Mistral AI models,
  Mistral API, implementing chat completions, streaming, function calling,
  structured outputs, code generation, OCR, or needing current model specs.
  Covers REST API usage via PHP cURL proxy, model selection, and capabilities.
---

# Mistral API Development Skill

You are an expert at integrating Mistral AI API into web applications using PHP proxy and vanilla JavaScript.

## Overview

The Mistral API provides access to open-weight and proprietary AI models:
- **Chat Completion** — text generation, conversation, reasoning
- **Streaming** — real-time token-by-token output via SSE
- **Function/Tool Calling** — let models invoke your functions
- **Structured Output** — JSON mode and JSON schema
- **Code Generation** — Codestral for FIM and code completion
- **Document AI / OCR** — text extraction from PDFs and images (Mistral OCR 3)
- **Speech-to-Text** — Voxtral Transcribe 2, real-time + batch, 13 languages
- **Embeddings** — text and code embeddings
- **Agents API** — multi-agent workflows with memory and connectors

## Current Mistral Models

- `mistral-large-2512`: flagship MoE, 41B active params, 256K context, multimodal
- `mistral-medium-2508`: frontier multimodal model
- `mistral-small-2506`: fast & efficient
- `codestral-2501`: code specialist, 80+ languages, FIM support, 256K context
- `devstral-2512`: agentic coding model + CLI agent (v2)
- `ministral-3b-2512` / `ministral-8b-2512` / `ministral-14b-2512`: compact edge models, text + vision
- `magistral-medium-1.2`: multimodal reasoning model
- `magistral-small-1.2`: lightweight reasoning model
- `voxtral-mini-transcribe-2`: batch speech-to-text, diarization
- `voxtral-realtime`: live streaming STT, 13 languages, open-source (Apache 2.0)
- `mistral-embed`: embedding model

> [!IMPORTANT]
> Mistral model IDs include release date (YYMM). Always use latest versions.
> Open-weight models can also be self-hosted.

## API Base

```
https://api.mistral.ai/v1/
```

Authentication: `Authorization: Bearer $MISTRAL_API_KEY`

## Key Endpoints

| Endpoint | Purpose |
|----------|---------|
| `/v1/chat/completions` | Chat completion (main) |
| `/v1/fim/completions` | Fill-in-the-middle (Codestral) |
| `/v1/embeddings` | Text/code embeddings |
| `/v1/agents/completions` | Agents API |
| `/v1/ocr` | Document AI / OCR |
| `/v1/audio/transcriptions` | Speech-to-text (Voxtral) |

## Quick Start (PHP)

```php
<?php
$payload = [
    'model' => 'mistral-small-2506',
    'messages' => [
        ['role' => 'system', 'content' => 'You are helpful.'],
        ['role' => 'user', 'content' => $userMessage]
    ],
    'temperature' => 0.7
];

$ch = curl_init('https://api.mistral.ai/v1/chat/completions');
curl_setopt_array($ch, [
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => [
        'Authorization: Bearer ' . getenv('MISTRAL_API_KEY'),
        'Content-Type: application/json'
    ],
    CURLOPT_POSTFIELDS => json_encode($payload),
    CURLOPT_RETURNTRANSFER => true
]);
$response = json_decode(curl_exec($ch), true);
curl_close($ch);
```

## Tool Calling

```php
$payload['tools'] = [[
    'type' => 'function',
    'function' => [
        'name' => 'get_weather',
        'description' => 'Get weather for a city',
        'parameters' => [
            'type' => 'object',
            'properties' => [
                'city' => ['type' => 'string']
            ],
            'required' => ['city']
        ]
    ]
]];
// Same response format as OpenAI: choices[0].message.tool_calls[]
```

## Structured Output

```php
// JSON mode
$payload['response_format'] = ['type' => 'json_object'];

// JSON schema (strict)
$payload['response_format'] = [
    'type' => 'json_schema',
    'json_schema' => [
        'name' => 'result',
        'schema' => [
            'type' => 'object',
            'properties' => [...],
            'required' => [...]
        ]
    ]
];
```

## Fill-in-the-Middle (Codestral)

```php
$payload = [
    'model' => 'codestral-2501',
    'prompt' => 'function fibonacci(',
    'suffix' => '}\nconsole.log(fibonacci(10));'
];
// POST to /v1/fim/completions
```

## API Docs (source of truth)

**llms.txt URL**: `https://docs.mistral.ai/llms.txt`

Key documentation pages:

- [Models Overview](https://docs.mistral.ai/getting-started/models/models_overview)
- [Chat Completions](https://docs.mistral.ai/capabilities/completion)
- [Function Calling](https://docs.mistral.ai/capabilities/function_calling)
- [Structured Output](https://docs.mistral.ai/capabilities/structured_output)
- [Coding](https://docs.mistral.ai/capabilities/coding)
- [Agents Introduction](https://docs.mistral.ai/agents/agents_introduction)
- [Document AI](https://docs.mistral.ai/capabilities/document_ai/document_ai_overview)
- [Embeddings](https://docs.mistral.ai/capabilities/embeddings/embeddings_overview)

## Related Skills
- `ai-api` — general AI integration patterns (streaming, agent loops, PHP proxy)
- `system-prompt-master` — prompt engineering
