---
name: javascript-master
description: >
  Write modern, clean, and error-free JavaScript code. Use when working with
  JS modules, DOM manipulation, event handling, or ES2026+ features.
---

# JavaScript Master

You are an absolute master of modern JavaScript. Elegant, efficient, and reliable code.

## Principles

**ES6+ ALWAYS.** Arrow functions, template literals, destructuring, spread, optional chaining, nullish coalescing.

**MODULES.** `type="module"` required. Import/export. No global variables. DRY — extract repeating code. **See `modular-code.md` for folder structure and granularity.**

**NULL/UNDEFINED HANDLING.** ALWAYS optional chaining (`?.`), nullish coalescing (`??`), default values. NEVER assume it exists.

**DOM.** QuerySelector, event delegation. NEVER hundreds of event listeners. Data attributes for state.

**NO CONSOLE ERRORS.** Code runs clean. No undefined/type errors.

**ES2026 FEATURES.** Prefer these when available:
- `Temporal` API — replaces `Date` for modern date/time handling
- `using` / `await using` — automatic resource cleanup (file handles, connections)
- `Array.fromAsync()` — create arrays from async iterables
- `Error.isError()` — reliable error type checking
- View Transitions API — smooth page state transitions (`document.startViewTransition()`)
- Popover API — native popovers with built-in focus management

## Async patterns

**ASYNC/AWAIT standard.** Try/catch for error handling. NEVER unhandled promise rejections.

```javascript
// Correct — async with error handling
async function fetchData(url) {
    try {
        const response = await fetch(url);
        if (!response.ok) throw new Error(`HTTP ${response.status}`);
        return await response.json();
    } catch (error) {
        console.error(`[fetchData] ${url}:`, error.message);
        return null;
    }
}
```

**PARALLEL REQUESTS.** `Promise.all` for independent requests. `Promise.allSettled` when some may fail.

```javascript
// Parallel fetch
const [users, settings] = await Promise.all([
    fetchData('/api/users'),
    fetchData('/api/settings')
]);

// Tolerant — continue even if some fail
const results = await Promise.allSettled([
    fetchData('/api/a'),
    fetchData('/api/b')
]);
const successful = results.filter(r => r.status === 'fulfilled').map(r => r.value);
```

**ABORT CONTROLLER.** Timeout and cancel for fetch requests.

```javascript
const controller = new AbortController();
const timeout = setTimeout(() => controller.abort(), 5000);

try {
    const response = await fetch(url, { signal: controller.signal });
    clearTimeout(timeout);
    return await response.json();
} catch (error) {
    if (error.name === 'AbortError') console.warn('Request timed out');
    throw error;
}
```

## Error handling patterns

**BOUNDARY ERRORS.** Catch errors at boundaries (API calls, user input, DOM operations), not inside pure functions.

```javascript
// Wrapper for safe DOM operations
function safeQuerySelector(selector, parent = document) {
    const el = parent.querySelector(selector);
    if (!el) console.warn(`[DOM] Element not found: ${selector}`);
    return el;
}
```

**CUSTOM ERRORS.** For distinguishing error types.

```javascript
class ApiError extends Error {
    constructor(status, message) {
        super(message);
        this.name = 'ApiError';
        this.status = status;
    }
}
```

## Event patterns

**DELEGATION.** One listener on parent, not hundreds on children.

```javascript
document.querySelector('.list').addEventListener('click', (e) => {
    const item = e.target.closest('[data-action]');
    if (!item) return;
    
    const action = item.dataset.action;
    const id = item.dataset.id;
    handleAction(action, id);
});
```

**CLEANUP.** AbortController for bulk listener cleanup.

```javascript
const ac = new AbortController();
element.addEventListener('click', handler, { signal: ac.signal });
element.addEventListener('keydown', handler2, { signal: ac.signal });
// Cleanup all at once:
ac.abort();
```

## Mindset

Before every line: Null/undefined? Event handler cleaned up? Async can fail? Reusable as module?

## Result
- Modern — ES6+ syntax
- Clean — clear structure, single responsibility
- Safe — handled null/undefined, handled errors
- Efficient — no memory leaks

## Related Skills
- `ai-api` — AI API integration (streaming, fetch, AbortController)
- `clean-code` — code quality principles
