---
name: elevenlabs-tts
description: >
  Use this skill when integrating ElevenLabs TTS API for text-to-speech,
  voice cloning, streaming audio, or voice generation.
  Covers REST API via PHP proxy, models, voices, and streaming.
---

# ElevenLabs TTS Skill

You are an expert at integrating ElevenLabs TTS API into web applications using PHP proxy and vanilla JavaScript.

## Overview

ElevenLabs provides ultra-realistic text-to-speech:
- **Multiple models** — quality vs latency tradeoffs
- **5000+ voices** — library, community, custom
- **Voice cloning** — instant and professional
- **Streaming** — WebSocket and HTTP streaming
- **70+ languages** — multilingual support
- **48kHz sample rate** — natural breath, micro-hesitations

## Current Models

| Model | Latency | Languages | Use Case |
|-------|---------|-----------|----------|
| `eleven_v3` | ~300ms | 70+ | Most expressive, dramatic delivery |
| `eleven_flash_v2_5` | ~75ms | 32 | Conversational, real-time |
| `eleven_turbo_v2_5` | ~250ms | 32 | Balanced quality/speed |
| `eleven_multilingual_v2` | ~300ms | 29 | Long-form, consistent |

> [!IMPORTANT]
> For real-time conversational use, prefer `eleven_flash_v2_5`.
> For best quality narration, use `eleven_v3`.

## API Base

```
https://api.elevenlabs.io/v1/
```

Authentication: `xi-api-key: $ELEVENLABS_API_KEY`

## Key Endpoints

| Endpoint | Purpose |
|----------|---------|
| `/v1/text-to-speech/{voice_id}` | Generate audio (returns binary) |
| `/v1/text-to-speech/{voice_id}/stream` | Stream audio chunks |
| `/v1/voices` | List available voices |
| `/v1/models` | List available models |
| `/v1/voice-generation/generate-voice` | Voice from prompt |

## Quick Start (PHP)

```php
<?php
$voiceId = 'pNInz6obpgDQGcFmaJgB';  // Adam voice
$payload = [
    'text' => $text,
    'model_id' => 'eleven_flash_v2_5',
    'voice_settings' => [
        'stability' => 0.5,
        'similarity_boost' => 0.75,
        'style' => 0.0,
        'use_speaker_boost' => true
    ]
];

$ch = curl_init("https://api.elevenlabs.io/v1/text-to-speech/{$voiceId}");
curl_setopt_array($ch, [
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => [
        'xi-api-key: ' . getenv('ELEVENLABS_API_KEY'),
        'Content-Type: application/json',
        'Accept: audio/mpeg'
    ],
    CURLOPT_POSTFIELDS => json_encode($payload),
    CURLOPT_RETURNTRANSFER => true
]);
$audioData = curl_exec($ch);
curl_close($ch);

header('Content-Type: audio/mpeg');
echo $audioData;
```

## Streaming Audio (PHP proxy)

```php
<?php
header('Content-Type: audio/mpeg');
header('Transfer-Encoding: chunked');

$ch = curl_init("https://api.elevenlabs.io/v1/text-to-speech/{$voiceId}/stream");
curl_setopt_array($ch, [
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => [
        'xi-api-key: ' . getenv('ELEVENLABS_API_KEY'),
        'Content-Type: application/json'
    ],
    CURLOPT_POSTFIELDS => json_encode($payload),
    CURLOPT_WRITEFUNCTION => function($ch, $data) {
        echo $data;
        ob_flush();
        flush();
        return strlen($data);
    }
]);
curl_exec($ch);
```

## Voice Settings Parameters

| Parameter | Range | Description |
|-----------|-------|-------------|
| `stability` | 0.0–1.0 | Higher = more consistent, lower = more varied |
| `similarity_boost` | 0.0–1.0 | Higher = closer to original voice |
| `style` | 0.0–1.0 | Style exaggeration (increases latency) |
| `use_speaker_boost` | bool | Enhance voice clarity |

## JS Audio Playback

```javascript
const response = await fetch('/api/tts-elevenlabs.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ text, voiceId, model: 'eleven_flash_v2_5' })
});
const blob = await response.blob();
const audio = new Audio(URL.createObjectURL(blob));
audio.play();
```

## Output Formats

- `mp3_44100_128` (default)
- `mp3_22050_32` (smaller file)
- `pcm_16000`, `pcm_22050`, `pcm_24000`, `pcm_44100`
- `ulaw_8000` (telephony)
- `opus` (various bitrates)

Set via query parameter: `?output_format=mp3_44100_128`

## API Docs

- [Text-to-Speech](https://elevenlabs.io/docs/api-reference/text-to-speech)
- [Voices](https://elevenlabs.io/docs/api-reference/voices)
- [Models](https://elevenlabs.io/docs/api-reference/models)
- [Voice Cloning](https://elevenlabs.io/docs/guides/voice-cloning)
- [Streaming](https://elevenlabs.io/docs/api-reference/streaming)

## Related Skills
- `ai-api` — AI integration patterns (LLM + TTS), PHP proxy architecture
- `tts-voice-instructor` — voice instruction engineering (OpenAI TTS)
