---
name: ai-api
description: >
  You are an expert in integrating AI APIs into web applications without build tools.
  Use when implementing LLM chat completion, streaming, function/tool calling, agent loops,
  TTS/text-to-speech, voice synthesis, voice cloning, and structured outputs using
  plain JavaScript + PHP proxy + JSON/MariaDB storage.
---

# AI API Integration (No-Build)

Expert skill for implementing AI provider APIs (LLM + TTS) in a vanilla JS + PHP stack.

> [!IMPORTANT]
> For prompt engineering (system prompts, temperature, instructions) → see skill `system-prompt-master`.
> For LLM provider-specific API details → see skills:
> `openai-api-dev`, `anthropic-api-dev`, `gemini-api-dev`, `mistral-api-dev`, `groq-api-dev`, `openrouter-api-dev`
> For TTS provider-specific details → see skills:
> `tts-voice-instructor`, `elevenlabs-tts`, `cartesia-tts`, `fish-audio-tts`, `playht-tts`
> For database schema → see skill `sql-mariadb`.

## Architecture

```
Browser (vanilla JS)         PHP Proxy              LLM Provider
────────────────────    ──────────────────    ─────────────────
fetch('/api/*.php')  →  cURL / stream       →  OpenAI / Gemini / Mistral
ReadableStream       ←  SSE passthrough     ←  Streaming chunks
```

### Security Rules
- API keys **ONLY in PHP** — NEVER expose to frontend
- PHP proxy validates input before forwarding
- Rate limiting on PHP endpoints
- Sanitize all LLM output before rendering (XSS prevention)

---

## Chat Completion

### PHP Endpoint Template
```php
<?php
header('Content-Type: application/json');
$input = json_decode(file_get_contents('php://input'), true);

$payload = [
    'model' => $input['model'] ?? 'gpt-4.1-mini',
    'messages' => $input['messages'],
    'temperature' => $input['temperature'] ?? 0.7
];

$ch = curl_init('https://api.openai.com/v1/chat/completions');
curl_setopt_array($ch, [
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => [
        'Authorization: Bearer ' . getenv('OPENAI_API_KEY'),
        'Content-Type: application/json'
    ],
    CURLOPT_POSTFIELDS => json_encode($payload),
    CURLOPT_RETURNTRANSFER => true
]);
$response = curl_exec($ch);
curl_close($ch);
echo $response;
```

### JS Client
```javascript
const response = await fetch('/api/chat.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ messages, model })
});
const data = await response.json();
```

---

## Streaming

### PHP Streaming Proxy
```php
<?php
header('Content-Type: text/event-stream');
header('Cache-Control: no-cache');
header('X-Accel-Buffering: no');

$payload['stream'] = true;

$ch = curl_init($apiUrl);
curl_setopt_array($ch, [
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => $headers,
    CURLOPT_POSTFIELDS => json_encode($payload),
    CURLOPT_WRITEFUNCTION => function($ch, $data) {
        echo $data;
        ob_flush();
        flush();
        return strlen($data);
    }
]);
curl_exec($ch);
curl_close($ch);
```

### JS Stream Reader
```javascript
async function streamChat(messages, onChunk) {
    const controller = new AbortController();
    const response = await fetch('/api/chat-stream.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ messages, stream: true }),
        signal: controller.signal
    });

    const reader = response.body.getReader();
    const decoder = new TextDecoder();

    while (true) {
        const { done, value } = await reader.read();
        if (done) break;
        const text = decoder.decode(value, { stream: true });
        // Parse SSE: each line starts with "data: "
        for (const line of text.split('\n')) {
            if (!line.startsWith('data: ') || line === 'data: [DONE]') continue;
            const chunk = JSON.parse(line.slice(6));
            const content = chunk.choices?.[0]?.delta?.content;
            if (content) onChunk(content);
        }
    }
    return controller; // for abort
}
```

> [!CAUTION]
> Use `ReadableStream` + `getReader()`, NOT `EventSource` (requires GET).
> Always provide `AbortController` for cancel capability.

---

## Tools / Function Calling

### Tool Definition (JSON Schema)
```javascript
const tools = [{
    type: 'function',
    function: {
        name: 'get_weather',
        description: 'Get current weather for a location',
        parameters: {
            type: 'object',
            properties: {
                location: { type: 'string', description: 'City name' },
                units: { type: 'string', enum: ['metric', 'imperial'] }
            },
            required: ['location']
        }
    }
}];
```

### PHP Agent Loop
```php
function agentLoop(array $messages, array $tools, int $maxIterations = 10): string {
    for ($i = 0; $i < $maxIterations; $i++) {
        $response = callLLM($messages, $tools);
        $choice = $response['choices'][0];

        // No tool calls → return final answer
        if ($choice['finish_reason'] !== 'tool_calls') {
            return $choice['message']['content'];
        }

        // Process tool calls
        $messages[] = $choice['message']; // assistant message with tool_calls
        foreach ($choice['message']['tool_calls'] as $call) {
            $fn = $call['function']['name'];
            $args = json_decode($call['function']['arguments'], true);
            $result = executeFunction($fn, $args);
            $messages[] = [
                'role' => 'tool',
                'tool_call_id' => $call['id'],
                'content' => json_encode($result)
            ];
        }
    }
    return 'Max iterations reached';
}
```

### JS-Side Tool Execution (Alternative)
When tools need browser access (DOM, geolocation, local data):
1. PHP returns `tool_calls` to JS
2. JS executes locally
3. JS sends results back to PHP
4. PHP continues agent loop with tool results

---

## Structured Output (JSON Schema)

```php
$payload['response_format'] = [
    'type' => 'json_schema',
    'json_schema' => [
        'name' => 'analysis_result',
        'strict' => true,
        'schema' => [
            'type' => 'object',
            'properties' => [
                'summary' => ['type' => 'string'],
                'score' => ['type' => 'integer', 'minimum' => 0, 'maximum' => 100],
                'tags' => ['type' => 'array', 'items' => ['type' => 'string']]
            ],
            'required' => ['summary', 'score', 'tags']
        ]
    ]
];
```

---

## TTS (Text-to-Speech)

```php
<?php
// api/tts.php
$payload = [
    'model' => 'gpt-4o-mini-tts',
    'input' => $input['text'],
    'voice' => $input['voice'] ?? 'alloy',
    'instructions' => $input['instructions'] ?? ''
];
// Returns audio binary → proxy to browser
header('Content-Type: audio/mpeg');
// ... curl with CURLOPT_WRITEFUNCTION for streaming audio
```

```javascript
// JS playback
const response = await fetch('/api/tts.php', {
    method: 'POST',
    body: JSON.stringify({ text, voice: 'nova' })
});
const blob = await response.blob();
const audio = new Audio(URL.createObjectURL(blob));
audio.play();
```

> For voice instruction engineering → see skill `tts-voice-instructor`.

---

## Data Storage

### Variant 1: JSON Files (Current)
```
data/conversations/{session_id}.json   — message history
data/tools/{session_id}_results.json   — tool execution results
```

```php
function saveMessage(string $sessionId, array $message): void {
    $path = "data/conversations/{$sessionId}.json";
    $history = file_exists($path) ? json_decode(file_get_contents($path), true) : [];
    $history[] = $message;
    file_put_contents($path, json_encode($history, JSON_PRETTY_PRINT));
}
```

### Variant 2: MariaDB (Future)
```sql
CREATE TABLE conversations (
    id INT AUTO_INCREMENT PRIMARY KEY,
    session_id VARCHAR(36) NOT NULL,
    role ENUM('system','user','assistant','tool') NOT NULL,
    content TEXT,
    tool_calls JSON,
    tool_call_id VARCHAR(64),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_session (session_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
```

→ For MariaDB patterns see skill `sql-mariadb`.

---

## Provider Compatibility

| Feature | OpenAI | Anthropic | Gemini | Mistral | Groq | OpenRouter |
|---------|--------|-----------|--------|---------|------|------------|
| Chat Completion | ✅ | ✅ (Messages API) | ✅ | ✅ | ✅ | ✅ |
| Streaming | ✅ SSE | ✅ SSE (custom events) | ✅ SSE | ✅ SSE | ✅ SSE | ✅ SSE |
| Tools/Function Calling | ✅ | ✅ (tool_use blocks) | ✅ | ✅ | ✅ | ✅ |
| JSON Schema Output | ✅ strict | ❌ | ✅ | ✅ | ❌ | model-dependent |
| TTS | ✅ | ❌ | ❌ | ❌ | ✅ (Whisper) | ❌ |
| Vision | ✅ | ✅ | ✅ | ✅ (Pixtral) | ❌ | model-dependent |
| OpenAI-compatible | ✅ | ❌ | ❌ | ✅ | ✅ | ✅ |

> [!WARNING]
> **Anthropic** uses its own Messages API format — NOT OpenAI-compatible.
> See `anthropic-api-dev` skill for format differences.

## Error Handling

```php
function callLLMWithRetry(array $payload, int $maxRetries = 3): array {
    for ($i = 0; $i < $maxRetries; $i++) {
        $response = callLLM($payload);
        if ($response !== null) return $response;
        usleep(pow(2, $i) * 100000); // exponential backoff
    }
    throw new RuntimeException('LLM API failed after retries');
}
```

## Checklist
- [ ] API keys in PHP env, never in JS
- [ ] Streaming uses ReadableStream (not EventSource)
- [ ] AbortController for cancel
- [ ] Agent loop has max iteration limit
- [ ] Tool definitions use proper JSON Schema
- [ ] Error handling with retry + backoff
- [ ] XSS sanitization of LLM output

## Related Skills
- `php-master` — PHP security, cURL, endpoint templates
- `javascript-master` — ES6+, async/await, fetch, AbortController
- `sql-mariadb` — MariaDB storage for conversations
- `system-prompt-master` — prompt engineering
