---
trigger: model_decision
description: When writing, editing, or reviewing PHP code -- critical syntax rules, tag encoding, output buffering, strict comparison, and common pitfalls that cause silent failures.
---

# PHP Syntax

## Critical rules
- ALWAYS use `<?php` (not HTML entity `&lt;?php`)
- No spaces before `<?php`
- First line of file = `<?php`

## API endpoint structure
```php
<?php
error_reporting(0);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method Not Allowed']);
    exit;
}

try {
    // Logic
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}
```

## Security
- Input validation: `file_get_contents('php://input')`
- Sanitization: `str_replace(['/', '\\', "\0"], '', $filename)`
- Path traversal: check `realpath()` against allowed dir
- Permissions: `is_writable($dir)`

## Headers
```php
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
```

## Response format
```php
// Success
echo json_encode(['success' => true, 'data' => $result]);

// Error
http_response_code(400);
echo json_encode(['success' => false, 'error' => 'Message']);
```

## Checklist
- Correct PHP tag `<?php`
- Headers before output
- Error handling (try/catch)
- Input validation
- Security (path traversal, XSS)
- HTTP status codes
- JSON response
- CORS headers

## Reference
- For complete PHP standards, use skill `php-master`
- For database operations, use skill `sql-mariadb`