---
trigger: always_on
description: Always active — modular code architecture for all technologies
---

# Modular Code Architecture

> For code quality across technologies see skill `clean-code`.

> [!CAUTION]
> **NEVER CREATE MONOLITHIC FILES!**
> Large files prevent efficient LLM agent work due to limited context window.

## Why Modularity

1. **LLM context** — agent loads only needed modules, not thousands of lines
2. **Maintenance** — changing one function = editing one file
3. **Testing** — isolated units are easier to test
4. **Parallel work** — multiple changes without conflicts

---

## Universal Principles

### Single Responsibility
- **One function = one purpose**
- **One file = one logical unit**
- If you describe a file using "and" or "also", split it

### Explicit Dependencies
- All imports at the beginning of the file
- No global variables between modules
- Dependency injection instead of hardcoded dependencies

### Naming
- File name = main export (`formatDate.js` -> `export function formatDate()`)
- Folders = logical groups (`/utils`, `/components`, `/services`)

---

## JavaScript / TypeScript

> For code quality see skill `javascript-master`.

### Structure
```
/js
├── main.js              # Entry point, initialization, event listeners
├── state.js             # Central state (singleton)
├── constants.js         # Global constants
├── /components          # UI components
├── /services            # API calls, external services
├── /utils               # Utility functions (formatters, validators)
├── /dom                 # DOM manipulation
└── /[feature]           # Feature-based folders
```

### Rules
- **Entry point** (`main.js`) — only imports, init, event listeners
- **One exported function/class per file** (except related helpers)
- **ES Modules** — `import`/`export`, no global variables
- **Dynamic imports** for lazy loading larger modules

### Correct Structure Example
```javascript
// BAD: app.js (2900 lines)
function sendMessage() { ... }
function formatDate() { ... }
function renderUI() { ... }
// ... 100 more functions

// GOOD: modular structure
// js/agent/send-message.js
import { state } from '../state.js';
import { formatMessage } from '../utils/format-message.js';
export async function sendMessage() { ... }

// js/utils/format-message.js
export function formatMessage(text) { ... }
```

---

## PHP

> For code quality see skill `php-master`, for database see skill `sql-mariadb`.

### Structure
```
/api
├── /endpoints           # Individual API endpoints
├── /services            # Business logic
├── /utils               # Helper functions
└── /middleware          # Auth, validation, error handling
```

### Rules
- **One class per file** (PSR-4 autoloading)
- **Endpoints** — thin, call services
- **Services** — business logic, testable
- **Namespace** matches folder structure

---

## CSS

> For code quality see skill `css-master`.

### Structure
```
/css
├── styles.css           # Import all components
├── /base                # Reset, typography, variables
├── /components          # Individual components
├── /layouts             # Layout patterns
└── /utilities           # Utility classes
```

### Rules
- **One file per component** (`/components/button.css`)
- **CSS custom properties** for repeated values
- **@import** or CSS @layer for organization
- **BEM or utility-first** — consistent naming

---

## When Larger Files Are OK

Exceptions where more code in one file is acceptable:

1. **Data files** — configuration, constants, definitions
2. **Generated code** — build output, bundled code
3. **Single complex algorithm** — one complex function with helper methods
4. **Legacy code** — during gradual refactoring

> [!IMPORTANT]
> Even in these cases, prefer splitting if it makes sense.

---

## Refactoring Checklist

When creating new code or modifying existing:

- [ ] Is each file under ~200 lines? (flexible, depends on context)
- [ ] Does each file have one clear purpose?
- [ ] Are all dependencies explicit (imports)?
- [ ] Can the file be tested in isolation?
- [ ] Is naming clear and consistent?

---

## Anti-patterns

### God Object
One file/class does everything.

### Copy-Paste
Duplicated code instead of shared function.

### Deep Nesting
10 levels of nested folders — max 3-4 levels.

### Circular Dependencies
Module A imports B, B imports A.

### Mixed Concerns
UI logic mixed with business logic and API calls.