"""
AI Usage UI - Uživatelské rozhraní pro zobrazování statistik a správu usage dat.
"""

import xbmc
import xbmcgui
import xbmcplugin
import xbmcaddon
from wcs.ai.UsageTracker import (
    get_usage_summary, reset_usage_data, format_cost, format_tokens, get_call_history
)
from wcs import utils

def display_ai_usage_menu(addon, handle):
    """Zobrazí hlavní menu pro AI usage statistiky."""
    xbmcplugin.setPluginCategory(handle, "AI Usage Statistiky")
    xbmcplugin.setContent(handle, 'files')
    
    # Načtení summary dat
    try:
        summary = get_usage_summary()
    except Exception as e:
        summary = {"error": str(e)}
    
    # Vždy zobrazíme základní menu, i když jsou data prázdná nebo chybná
    if "error" in summary:
        # Zobrazíme chybu ale pokračujeme v zobrazení menu
        listitem = xbmcgui.ListItem(f"PROBLEM s daty: {summary['error']}")
        listitem.setArt({'thumb': 'DefaultAddonService.png'})
        utils.set_video_info_tag(listitem, {'plot': 'Chyba při načítání usage dat'})
        xbmcplugin.addDirectoryItem(handle, "", listitem, False)
        
        # Nastavíme prázdná data pro zbytek funkce
        summary = {
            "totals": {"cost": 0.0, "requests": 0, "avg_cost_per_request": 0.0},
            "today": {"cost": 0.0, "requests": 0},
            "this_month": {"cost": 0.0, "requests": 0},
            "providers": {},
            "top_models": []
        }
    
    # Celkové statistiky
    totals = summary.get("totals", {})
    total_cost = totals.get("cost", 0.0)
    total_requests = totals.get("requests", 0)
    avg_cost = totals.get("avg_cost_per_request", 0.0)
    
    # Dnešní statistiky  
    today = summary.get("today", {})
    today_cost = today.get("cost", 0.0)
    today_requests = today.get("requests", 0)
    
    # Měsíční statistiky
    month = summary.get("this_month", {})
    month_cost = month.get("cost", 0.0)
    month_requests = month.get("requests", 0)
    
    # 1. Poslední použití - historie individuálních volání
    listitem = xbmcgui.ListItem("Poslední použití")
    listitem.setArt({'thumb': 'DefaultAddonContextItem.png'})
    utils.set_video_info_tag(listitem, {'plot': 'Historie individuálních AI API volání s detailními informacemi'})
    url = f"plugin://{addon.getAddonInfo('id')}?action=ai_usage_details&type=call_history"
    xbmcplugin.addDirectoryItem(handle, url, listitem, True)
    
    # 2. Celkový přehled - s token breakdown
    total_input_tokens = sum(sum(m.get("input_tokens", 0) for m in p.get("models", {}).values()) for p in summary.get("providers", {}).values())
    total_cached_tokens = sum(sum(m.get("cached_tokens", 0) for m in p.get("models", {}).values()) for p in summary.get("providers", {}).values())
    total_output_tokens = sum(sum(m.get("output_tokens", 0) for m in p.get("models", {}).values()) for p in summary.get("providers", {}).values())
    
    token_info = f"in:{format_tokens(total_input_tokens)}"
    if total_cached_tokens > 0:
        token_info += f" cached:{format_tokens(total_cached_tokens)}"
    token_info += f" out:{format_tokens(total_output_tokens)}"
    
    listitem = xbmcgui.ListItem("Celkový přehled")
    listitem.setArt({'thumb': 'DefaultAddonNone.png'})
    utils.set_video_info_tag(listitem, {'plot': f'Celkové náklady: {format_cost(total_cost, addon)}\nCelkem požadavků: {total_requests}\nPrůměr na požadavek: {format_cost(avg_cost, addon)}\nTokeny: {token_info}'})
    url = f"plugin://{addon.getAddonInfo('id')}?action=ai_usage_details&type=overview"
    xbmcplugin.addDirectoryItem(handle, url, listitem, True)
    
    # 3. Dnešní statistiky
    listitem = xbmcgui.ListItem("Dnes")
    listitem.setArt({'thumb': 'DefaultAddonContextItem.png'})
    utils.set_video_info_tag(listitem, {'plot': f'Náklady dnes: {format_cost(today_cost, addon)}\nPožadavků dnes: {today_requests}'})
    url = f"plugin://{addon.getAddonInfo('id')}?action=ai_usage_details&type=today"
    xbmcplugin.addDirectoryItem(handle, url, listitem, True)
    
    # 4. Měsíční statistiky
    listitem = xbmcgui.ListItem("Tento měsíc")
    listitem.setArt({'thumb': 'DefaultAddonWebSkin.png'})
    utils.set_video_info_tag(listitem, {'plot': f'Náklady tento měsíc: {format_cost(month_cost, addon)}\nPožadavků tento měsíc: {month_requests}'})
    url = f"plugin://{addon.getAddonInfo('id')}?action=ai_usage_details&type=month"
    xbmcplugin.addDirectoryItem(handle, url, listitem, True)
    
    # 5. Breakdown podle poskytovatelů - s token breakdown
    providers = summary.get("providers", {})
    for provider, provider_data in providers.items():
        if provider_data.get("total_requests", 0) > 0:
            # Token breakdown pro poskytovatele
            provider_input = sum(m.get("input_tokens", 0) for m in provider_data.get("models", {}).values())
            provider_cached = sum(m.get("cached_tokens", 0) for m in provider_data.get("models", {}).values())
            provider_output = sum(m.get("output_tokens", 0) for m in provider_data.get("models", {}).values())
            
            token_info = f"in:{format_tokens(provider_input)}"
            if provider_cached > 0:
                token_info += f" cached:{format_tokens(provider_cached)}"
            token_info += f" out:{format_tokens(provider_output)}"
            
            listitem = xbmcgui.ListItem(f"{provider.upper()}")
            listitem.setArt({'thumb': 'DefaultAddonProgram.png'})
            utils.set_video_info_tag(listitem, {'plot': f'Náklady: {format_cost(provider_data.get("total_cost", 0.0), addon)}\nPožadavků: {provider_data.get("total_requests", 0)}\nTokeny: {token_info}'})
            url = f"plugin://{addon.getAddonInfo('id')}?action=ai_usage_details&type=provider&provider={provider}"
            xbmcplugin.addDirectoryItem(handle, url, listitem, True)
    
    # 6. Nejvíce používané modely - s token info
    top_models = summary.get("top_models", [])[:5]
    if top_models:
        listitem = xbmcgui.ListItem("Top 5 modelů podle nákladů")
        listitem.setArt({'thumb': 'DefaultAddonRepository.png'})
        utils.set_video_info_tag(listitem, {'plot': 'Nejpoužívanější modely seřazené podle nákladů s detaily o tokenech'})
        url = f"plugin://{addon.getAddonInfo('id')}?action=ai_usage_details&type=top_models"
        xbmcplugin.addDirectoryItem(handle, url, listitem, True)
    
    # Separátor
    listitem = xbmcgui.ListItem("--- SPRAVA ---")
    listitem.setArt({'thumb': 'DefaultAddonNone.png'})
    xbmcplugin.addDirectoryItem(handle, "", listitem, False)
    
    # 7. Možnosti resetu
    listitem = xbmcgui.ListItem("Správa dat")
    listitem.setArt({'thumb': 'DefaultAddonService.png'})
    utils.set_video_info_tag(listitem, {'plot': 'Možnosti mazání a resetu statistik'})
    url = f"plugin://{addon.getAddonInfo('id')}?action=ai_usage_reset_menu"
    xbmcplugin.addDirectoryItem(handle, url, listitem, True)
    
    # 8. Export dat
    listitem = xbmcgui.ListItem("Export statistik")
    listitem.setArt({'thumb': 'DefaultAddonNone.png'})
    utils.set_video_info_tag(listitem, {'plot': 'Export detailních statistik do textového formátu'})
    url = f"plugin://{addon.getAddonInfo('id')}?action=ai_usage_export"
    xbmcplugin.addDirectoryItem(handle, url, listitem, False)
    
    # 9. Pokud nejsou žádná data, zobrazíme informaci
    if total_requests == 0:
        listitem = xbmcgui.ListItem("Zatím žádná AI usage data")
        listitem.setArt({'thumb': 'DefaultAddonNone.png'})
        utils.set_video_info_tag(listitem, {'plot': 'Nebyly ještě provedeny žádné AI API požadavky. Použijte AI funkce v addonu a statistiky se zde zobrazí.'})
        xbmcplugin.addDirectoryItem(handle, "", listitem, False)
    
    xbmcplugin.endOfDirectory(handle)

def display_ai_usage_details(params, addon, handle):
    """Zobrazí detailní statistiky podle typu."""
    detail_type = params.get('type', 'overview')
    provider = params.get('provider', '')
    
    xbmcplugin.setPluginCategory(handle, f"AI Usage - {detail_type}")
    xbmcplugin.setContent(handle, 'files')
    
    summary = get_usage_summary()
    
    if "error" in summary:
        listitem = xbmcgui.ListItem(f"Chyba: {summary['error']}")
        listitem.setArt({'thumb': 'DefaultAddonService.png'})
        xbmcplugin.addDirectoryItem(handle, "", listitem, False)
        xbmcplugin.endOfDirectory(handle)
        return
    
    if detail_type == "overview":
        _display_overview_details(summary, addon, handle)
    elif detail_type == "today":
        _display_period_details(summary["today"], "Dnes", addon, handle)
    elif detail_type == "month":
        _display_period_details(summary["this_month"], "Tento měsíc", addon, handle)
    elif detail_type == "provider":
        _display_provider_details(summary["providers"].get(provider, {}), provider, addon, handle)
    elif detail_type == "top_models":
        _display_top_models(summary["top_models"], addon, handle)
    elif detail_type == "call_history":
        _display_call_history(addon, handle)
    
    xbmcplugin.endOfDirectory(handle)

def _display_overview_details(summary, addon, handle):
    """Zobrazí detailní celkový přehled."""
    totals = summary.get("totals", {})
    providers = summary.get("providers", {})
    
    # Celkové statistiky
    listitem = xbmcgui.ListItem(
        f"Celkové náklady: {format_cost(totals.get('cost', 0.0), addon)}"
    )
    listitem.setArt({'thumb': 'DefaultAddonNone.png'})
    xbmcplugin.addDirectoryItem(handle, "", listitem, False)
    
    listitem = xbmcgui.ListItem(
        f"Celkem požadavků: {totals.get('requests', 0)}"
    )
    listitem.setArt({'thumb': 'DefaultAddonNone.png'})
    xbmcplugin.addDirectoryItem(handle, "", listitem, False)
    
    listitem = xbmcgui.ListItem(
        f"Průměrné náklady na požadavek: {format_cost(totals.get('avg_cost_per_request', 0.0), addon)}"
    )
    listitem.setArt({'thumb': 'DefaultAddonNone.png'})
    xbmcplugin.addDirectoryItem(handle, "", listitem, False)
    
    # Detailní breakdown podle modelů
    all_models = []
    for provider, provider_data in providers.items():
        for model_name, model_data in provider_data.get("models", {}).items():
            all_models.append({
                "provider": provider,
                "model": model_name,
                "data": model_data
            })
    
    if all_models:
        listitem = xbmcgui.ListItem("─ Breakdown podle modelů ─")
        listitem.setArt({'thumb': 'DefaultAddonNone.png'})
        xbmcplugin.addDirectoryItem(handle, "", listitem, False)
        
        # Seřazení modelů podle nákladů
        sorted_models = sorted(all_models, key=lambda x: x["data"].get("total_cost", 0.0), reverse=True)
        
        for model_info in sorted_models:
            model_name = model_info["model"]
            provider = model_info["provider"] 
            model_data = model_info["data"]
            
            input_tokens = model_data.get("input_tokens", 0)
            cached_tokens = model_data.get("cached_tokens", 0)
            output_tokens = model_data.get("output_tokens", 0)
            
            label = (
                f"{provider.upper()}/{model_name} - "
                f"{format_cost(model_data.get('total_cost', 0.0), addon)} - "
                f"{model_data.get('requests', 0)}x"
            )
            
            listitem = xbmcgui.ListItem(label)
            listitem.setArt({'thumb': 'DefaultAddonProgram.png'})
            
            # Detailní popis do plot
            plot_parts = [
                f"Model: {model_name}",
                f"Poskytovatel: {provider.upper()}",
                f"Celkové náklady: {format_cost(model_data.get('total_cost', 0.0), addon)}",
                f"Počet požadavků: {model_data.get('requests', 0)}",
                f"Input tokeny: {format_tokens(model_data.get('input_tokens', 0))}",
                f"Output tokeny: {format_tokens(model_data.get('output_tokens', 0))}"
            ]
            
            if model_data.get("cached_tokens", 0) > 0:
                plot_parts.append(f"Cached tokeny: {format_tokens(model_data.get('cached_tokens', 0))}")
            
            avg_cost = model_data.get('total_cost', 0.0) / max(model_data.get('requests', 1), 1)
            plot_parts.append(f"Průměr na požadavek: {format_cost(avg_cost, addon)}")
            
            # Request types breakdown
            request_types = model_data.get("request_types", {})
            if request_types:
                plot_parts.append("\nTypy požadavků:")
                for req_type, type_data in request_types.items():
                    plot_parts.append(f"  {req_type}: {type_data.get('count', 0)}x - {format_cost(type_data.get('cost', 0.0), addon)}")
            
            utils.set_video_info_tag(listitem, {'plot': '\n'.join(plot_parts)})
            
            # Přidání URL pro kliknutí na model
            url = f"plugin://{addon.getAddonInfo('id')}?action=ai_usage_model_detail&provider={provider}&model={model_name}"
            xbmcplugin.addDirectoryItem(handle, url, listitem, True)

def _display_period_details(period_data, period_name, addon, handle):
    """Zobrazí detaily pro konkrétní období."""
    # Celkové statistiky pro období
    listitem = xbmcgui.ListItem(
        f"{period_name} - naklady: {format_cost(period_data.get('cost', 0.0), addon)}"
    )
    listitem.setArt({'thumb': 'DefaultAddonNone.png'})
    xbmcplugin.addDirectoryItem(handle, "", listitem, False)
    
    listitem = xbmcgui.ListItem(
        f"{period_name} - pozadavku: {period_data.get('requests', 0)}"
    )
    listitem.setArt({'thumb': 'DefaultAddonNone.png'})
    xbmcplugin.addDirectoryItem(handle, "", listitem, False)
    
    # Pro detailní model breakdown musíme přistoupit k plným datům
    # Získáme plná data a zobrazíme modely z celkového přehledu
    try:
        from wcs.ai.UsageTracker import get_usage_summary
        full_summary = get_usage_summary()
        all_providers = full_summary.get("providers", {})
        
        # Vytvoříme seznam všech modelů pro obecný přehled
        all_models = []
        for provider, provider_data in all_providers.items():
            for model_name, model_data in provider_data.get("models", {}).items():
                all_models.append({
                    "provider": provider,
                    "model": model_name,
                    "data": model_data
                })
        
        if all_models:
            listitem = xbmcgui.ListItem(f"─ Všechny použité modely ─")
            listitem.setArt({'thumb': 'DefaultAddonNone.png'})
            xbmcplugin.addDirectoryItem(handle, "", listitem, False)
            
            # Seřazení modelů podle nákladů
            sorted_models = sorted(all_models, key=lambda x: x["data"].get("total_cost", 0.0), reverse=True)
            
            for model_info in sorted_models:
                model_name = model_info["model"]
                provider = model_info["provider"] 
                model_data = model_info["data"]
                
                input_tokens = model_data.get("input_tokens", 0)
                cached_tokens = model_data.get("cached_tokens", 0)
                output_tokens = model_data.get("output_tokens", 0)
                
                label = (
                    f"{provider.upper()}/{model_name} - "
                    f"{format_cost(model_data.get('total_cost', 0.0), addon)} - "
                    f"{model_data.get('requests', 0)}x"
                )
                
                listitem = xbmcgui.ListItem(label)
                listitem.setArt({'thumb': 'DefaultAddonProgram.png'})
                
                # Detailní popis do plot
                plot_parts = [
                    f"Model: {model_name}",
                    f"Poskytovatel: {provider.upper()}",
                    f"Celkové náklady: {format_cost(model_data.get('total_cost', 0.0), addon)}",
                    f"Počet požadavků: {model_data.get('requests', 0)}",
                    f"Input tokeny: {format_tokens(model_data.get('input_tokens', 0))}",
                    f"Output tokeny: {format_tokens(model_data.get('output_tokens', 0))}"
                ]
                
                if model_data.get("cached_tokens", 0) > 0:
                    plot_parts.append(f"Cached tokeny: {format_tokens(model_data.get('cached_tokens', 0))}")
                
                avg_cost = model_data.get('total_cost', 0.0) / max(model_data.get('requests', 1), 1)
                plot_parts.append(f"Průměr na požadavek: {format_cost(avg_cost, addon)}")
                
                # Request types breakdown
                request_types = model_data.get("request_types", {})
                if request_types:
                    plot_parts.append("\nTypy požadavků:")
                    for req_type, type_data in request_types.items():
                        plot_parts.append(f"  {req_type}: {type_data.get('count', 0)}x - {format_cost(type_data.get('cost', 0.0), addon)}")
                
                utils.set_video_info_tag(listitem, {'plot': '\n'.join(plot_parts)})
                
                # Přidání URL pro kliknutí na model
                url = f"plugin://{addon.getAddonInfo('id')}?action=ai_usage_model_detail&provider={provider}&model={model_name}"
                xbmcplugin.addDirectoryItem(handle, url, listitem, True)
    
    except Exception as e:
        # Fallback pokud se nepodaří načíst data
        listitem = xbmcgui.ListItem("Nepodařilo se načíst detaily modelů")
        listitem.setArt({'thumb': 'DefaultAddonNone.png'})
        xbmcplugin.addDirectoryItem(handle, "", listitem, False)

def _display_provider_details(provider_data, provider_name, addon, handle):
    """Zobrazí detaily pro konkrétního poskytovatele."""
    # Celkové statistiky poskytovatele
    listitem = xbmcgui.ListItem(
        f"{provider_name.upper()} - celkové náklady: {format_cost(provider_data.get('total_cost', 0.0), addon)}"
    )
    listitem.setArt({'thumb': 'DefaultAddonNone.png'})
    xbmcplugin.addDirectoryItem(handle, "", listitem, False)
    
    listitem = xbmcgui.ListItem(
        f"{provider_name.upper()} - celkem požadavků: {provider_data.get('total_requests', 0)}"
    )
    listitem.setArt({'thumb': 'DefaultAddonNone.png'})
    xbmcplugin.addDirectoryItem(handle, "", listitem, False)
    
    # Breakdown podle modelů
    models = provider_data.get("models", {})
    if models:
        listitem = xbmcgui.ListItem("─ Breakdown podle modelů ─")
        listitem.setArt({'thumb': 'DefaultAddonNone.png'})
        xbmcplugin.addDirectoryItem(handle, "", listitem, False)
        
        # Seřazení modelů podle nákladů
        sorted_models = sorted(models.items(), key=lambda x: x[1].get("total_cost", 0.0), reverse=True)
        
        for model_name, model_data in sorted_models:
            input_tokens = model_data.get("input_tokens", 0)
            cached_tokens = model_data.get("cached_tokens", 0)
            output_tokens = model_data.get("output_tokens", 0)
            
            label = (
                f"{model_name} - "
                f"{format_cost(model_data.get('total_cost', 0.0), addon)} - "
                f"{model_data.get('requests', 0)}x"
            )
            
            listitem = xbmcgui.ListItem(label)
            listitem.setArt({'thumb': 'DefaultAddonProgram.png'})
            
            # Detailní popis do plot
            plot_parts = [
                f"Model: {model_name}",
                f"Celkové náklady: {format_cost(model_data.get('total_cost', 0.0), addon)}",
                f"Počet požadavků: {model_data.get('requests', 0)}",
                f"Input tokeny: {format_tokens(model_data.get('input_tokens', 0))}",
                f"Output tokeny: {format_tokens(model_data.get('output_tokens', 0))}"
            ]
            
            if model_data.get("cached_tokens", 0) > 0:
                plot_parts.append(f"Cached tokeny: {format_tokens(model_data.get('cached_tokens', 0))}")
            
            avg_cost = model_data.get('total_cost', 0.0) / max(model_data.get('requests', 1), 1)
            plot_parts.append(f"Průměr na požadavek: {format_cost(avg_cost, addon)}")
            
            # Request types breakdown
            request_types = model_data.get("request_types", {})
            if request_types:
                plot_parts.append("\nTypy požadavků:")
                for req_type, type_data in request_types.items():
                    plot_parts.append(f"  {req_type}: {type_data.get('count', 0)}x - {format_cost(type_data.get('cost', 0.0), addon)}")
            
            utils.set_video_info_tag(listitem, {'plot': '\n'.join(plot_parts)})
            
            # Přidání URL pro kliknutí na model
            url = f"plugin://{addon.getAddonInfo('id')}?action=ai_usage_model_detail&provider={provider_name}&model={model_name}"
            xbmcplugin.addDirectoryItem(handle, url, listitem, True)

def _display_top_models(top_models, addon, handle):
    """Zobrazí top modely podle nákladů s token statistikami."""
    for i, model_info in enumerate(top_models, 1):
        # Token breakdown
        input_tokens = model_info.get("input_tokens", 0)
        cached_tokens = model_info.get("cached_tokens", 0)
        output_tokens = model_info.get("output_tokens", 0)
        total_tokens = input_tokens + cached_tokens + output_tokens
        
        label = (
            f"{i}. {model_info['provider']}/{model_info['model']} - "
            f"{format_cost(model_info['cost'], addon)} - "
            f"{model_info['requests']}x"
        )
        
        listitem = xbmcgui.ListItem(label)
        listitem.setArt({'thumb': 'DefaultAddonProgram.png'})
        
        plot = (
            f"Pozice: #{i}\n"
            f"Poskytovatel: {model_info['provider']}\n"
            f"Model: {model_info['model']}\n"
            f"Celkové náklady: {format_cost(model_info['cost'], addon)}\n"
            f"Počet požadavků: {model_info['requests']}\n"
            f"Průměr na požadavek: {format_cost(model_info['avg_cost_per_request'], addon)}\n"
            f"Input tokeny: {format_tokens(model_info.get('input_tokens', 0))}\n"
            f"Cached tokeny: {format_tokens(model_info.get('cached_tokens', 0))}\n"
            f"Output tokeny: {format_tokens(model_info.get('output_tokens', 0))}\n"
            f"Celkem tokenů: {format_tokens(total_tokens)}"
        )
        
        utils.set_video_info_tag(listitem, {'plot': plot})
        
        # Přidání URL pro kliknutí na model
        url = f"plugin://{addon.getAddonInfo('id')}?action=ai_usage_model_detail&provider={model_info['provider']}&model={model_info['model']}"
        xbmcplugin.addDirectoryItem(handle, url, listitem, True)

def _display_call_history(addon, handle):
    """Zobrazí historii individuálních AI volání."""
    import datetime
    
    try:
        call_history = get_call_history(50)  # Posledních 50 volání
        
        if not call_history:
            listitem = xbmcgui.ListItem("Zatím žádná AI volání")
            listitem.setArt({'thumb': 'DefaultAddonNone.png'})
            utils.set_video_info_tag(listitem, {'plot': 'Nebyly ještě provedeny žádné AI API požadavky'})
            xbmcplugin.addDirectoryItem(handle, "", listitem, False)
            return
        
        # Výpočet celkové ceny za všechna volání
        total_cost = sum(call.get("cost", 0.0) for call in call_history)
        
        # Zobrazení počtu volání a celkové ceny
        listitem = xbmcgui.ListItem(f"Posledních {len(call_history)} použití AI - {format_cost(total_cost, addon)}")
        listitem.setArt({'thumb': 'DefaultAddonNone.png'})
        xbmcplugin.addDirectoryItem(handle, "", listitem, False)
        
        listitem = xbmcgui.ListItem("─────────────────")
        listitem.setArt({'thumb': 'DefaultAddonNone.png'})
        xbmcplugin.addDirectoryItem(handle, "", listitem, False)
        
        for call in call_history:
            # Parsování timestamp
            try:
                timestamp_str = call.get("timestamp", "")
                if timestamp_str:
                    dt = datetime.datetime.fromisoformat(timestamp_str.replace('Z', '+00:00'))
                    time_str = dt.strftime("%d.%m %H:%M:%S")
                else:
                    time_str = "Neznámý čas"
            except:
                time_str = "Neznámý čas"
            
            # Mapování typů požadavků na češtinu
            request_type_map = {
                "quiz": "KVÍZ",
                "game": "TEXTOVÁ HRA", 
                "movie_recommendation": "AI DOPORUČENÍ",
                "mytv_recommendation": "AI DOPORUČENÍ",
                "channel_name": "AI NÁZEV KANÁLU",
                "chat_recommendation": "AI CHAT",
                "millionaire_easy_questions": "MILIONÁŘ",
                "millionaire_medium_questions": "MILIONÁŘ",
                "millionaire_hard_questions": "MILIONÁŘ",
                "standard": "OSTATNÍ"
            }
            
            request_type_czech = request_type_map.get(call.get('request_type', 'standard'), 'OSTATNÍ')
            
            # Hlavní label - nový formát bez číslování a tokenů
            label = (
                f"{time_str} - {request_type_czech} - "
                f"{call.get('provider', '?').upper()}/{call.get('model', '?')} - "
                f"{format_cost(call.get('cost', 0.0), addon)}"
            )
            
            listitem = xbmcgui.ListItem(label)
            listitem.setArt({'thumb': 'DefaultAddonProgram.png'})
            
            # Detailní informace do plot - nový formát
            plot_parts = [
                f"Typ požadavku: {request_type_czech}",
                f"Poskytovatel: {call.get('provider', 'Neznámý').upper()}",
                f"Model: {call.get('model', 'Neznámý')}",
                f"Náklady: {format_cost(call.get('cost', 0.0), addon)}",
                f"Input tokeny: {format_tokens(call.get('input_tokens', 0))}",
                f"Output tokeny: {format_tokens(call.get('output_tokens', 0))}",
                f"Celkem tokenů: {format_tokens(call.get('total_tokens', 0))}"
            ]
            
            if call.get("cached_tokens", 0) > 0:
                plot_parts.insert(-1, f"Cached tokeny: {format_tokens(call.get('cached_tokens', 0))}")
            
            utils.set_video_info_tag(listitem, {'plot': '\n'.join(plot_parts)})
            xbmcplugin.addDirectoryItem(handle, "", listitem, False)
            
    except Exception as e:
        listitem = xbmcgui.ListItem(f"Chyba: {str(e)}")
        listitem.setArt({'thumb': 'DefaultAddonService.png'})
        xbmcplugin.addDirectoryItem(handle, "", listitem, False)

def display_ai_usage_model_detail(params, addon, handle):
    """Zobrazí detailní statistiky pro konkrétní model."""
    provider = params.get('provider', '')
    model = params.get('model', '')
    
    xbmcplugin.setPluginCategory(handle, f"AI Usage - {provider.upper()}/{model}")
    xbmcplugin.setContent(handle, 'files')
    
    try:
        summary = get_usage_summary()
        provider_data = summary.get("providers", {}).get(provider, {})
        model_data = provider_data.get("models", {}).get(model, {})
        
        if not model_data:
            listitem = xbmcgui.ListItem("Model nenalezen")
            listitem.setArt({'thumb': 'DefaultAddonService.png'})
            xbmcplugin.addDirectoryItem(handle, "", listitem, False)
            xbmcplugin.endOfDirectory(handle)
            return
        
        # Základní statistiky modelu
        input_tokens = model_data.get("input_tokens", 0)
        cached_tokens = model_data.get("cached_tokens", 0)
        output_tokens = model_data.get("output_tokens", 0)
        total_cost = model_data.get("total_cost", 0.0)
        requests = model_data.get("requests", 0)
        avg_cost = total_cost / max(requests, 1)
        
        listitem = xbmcgui.ListItem(f"Model: {model}")
        listitem.setArt({'thumb': 'DefaultAddonNone.png'})
        xbmcplugin.addDirectoryItem(handle, "", listitem, False)
        
        listitem = xbmcgui.ListItem(f"Poskytovatel: {provider.upper()}")
        listitem.setArt({'thumb': 'DefaultAddonNone.png'})
        xbmcplugin.addDirectoryItem(handle, "", listitem, False)
        
        listitem = xbmcgui.ListItem(f"Celkové náklady: {format_cost(total_cost, addon)}")
        listitem.setArt({'thumb': 'DefaultAddonNone.png'})
        xbmcplugin.addDirectoryItem(handle, "", listitem, False)
        
        listitem = xbmcgui.ListItem(f"Celkem požadavků: {requests}")
        listitem.setArt({'thumb': 'DefaultAddonNone.png'})
        xbmcplugin.addDirectoryItem(handle, "", listitem, False)
        
        listitem = xbmcgui.ListItem(f"Průměr na požadavek: {format_cost(avg_cost, addon)}")
        listitem.setArt({'thumb': 'DefaultAddonNone.png'})
        xbmcplugin.addDirectoryItem(handle, "", listitem, False)
        
        # Token statistiky
        listitem = xbmcgui.ListItem(f"Input tokeny: {format_tokens(input_tokens)}")
        listitem.setArt({'thumb': 'DefaultAddonNone.png'})
        xbmcplugin.addDirectoryItem(handle, "", listitem, False)
        
        if cached_tokens > 0:
            listitem = xbmcgui.ListItem(f"Cached tokeny: {format_tokens(cached_tokens)}")
            listitem.setArt({'thumb': 'DefaultAddonNone.png'})
            xbmcplugin.addDirectoryItem(handle, "", listitem, False)
        
        listitem = xbmcgui.ListItem(f"Output tokeny: {format_tokens(output_tokens)}")
        listitem.setArt({'thumb': 'DefaultAddonNone.png'})
        xbmcplugin.addDirectoryItem(handle, "", listitem, False)
        
        # Breakdown podle typů požadavků
        request_types = model_data.get("request_types", {})
        if request_types:
            listitem = xbmcgui.ListItem("─ Breakdown podle typu požadavků ─")
            listitem.setArt({'thumb': 'DefaultAddonNone.png'})
            xbmcplugin.addDirectoryItem(handle, "", listitem, False)
            
            # Seřazení podle nákladů
            sorted_types = sorted(request_types.items(), key=lambda x: x[1].get("cost", 0.0), reverse=True)
            
            for req_type, type_data in sorted_types:
                type_cost = type_data.get("cost", 0.0)
                type_count = type_data.get("count", 0)
                type_avg = type_cost / max(type_count, 1)
                
                label = (
                    f"{req_type} - "
                    f"{format_cost(type_cost, addon)} - "
                    f"{type_count}x - "
                    f"průměr: {format_cost(type_avg, addon)}"
                )
                
                listitem = xbmcgui.ListItem(label)
                listitem.setArt({'thumb': 'DefaultAddonProgram.png'})
                
                # Detailní popis pro typ požadavku
                plot_parts = [
                    f"Typ požadavku: {req_type}",
                    f"Model: {model}",
                    f"Poskytovatel: {provider.upper()}",
                    f"Náklady na tento typ: {format_cost(type_cost, addon)}",
                    f"Počet požadavků: {type_count}",
                    f"Průměr na požadavek: {format_cost(type_avg, addon)}",
                    f"Podíl z celkových nákladů: {(type_cost/max(total_cost, 0.01)*100):.1f}%"
                ]
                
                utils.set_video_info_tag(listitem, {'plot': '\n'.join(plot_parts)})
                xbmcplugin.addDirectoryItem(handle, "", listitem, False)
        
    except Exception as e:
        listitem = xbmcgui.ListItem(f"Chyba: {str(e)}")
        listitem.setArt({'thumb': 'DefaultAddonService.png'})
        xbmcplugin.addDirectoryItem(handle, "", listitem, False)
    
    xbmcplugin.endOfDirectory(handle)

def display_ai_usage_reset_menu(addon, handle):
    """Zobrazí menu pro reset a správu usage dat."""
    xbmcplugin.setPluginCategory(handle, "AI Usage - Správa dat")
    xbmcplugin.setContent(handle, 'files')
    
    # 1. Reset všech dat
    listitem = xbmcgui.ListItem("Resetovat všechna data")
    listitem.setArt({'thumb': 'DefaultAddonService.png'})
    utils.set_video_info_tag(listitem, {'plot': 'Smaže všechny statistiky AI usage - NEVRATNÁ AKCE!'})
    url = f"plugin://{addon.getAddonInfo('id')}?action=ai_usage_reset&period=all"
    xbmcplugin.addDirectoryItem(handle, url, listitem, False)
    
    # 2. Reset měsíčních dat
    listitem = xbmcgui.ListItem("Resetovat měsíční data")
    listitem.setArt({'thumb': 'DefaultAddonWebSkin.png'})
    utils.set_video_info_tag(listitem, {'plot': 'Smaže pouze měsíční statistiky'})
    url = f"plugin://{addon.getAddonInfo('id')}?action=ai_usage_reset&period=monthly"
    xbmcplugin.addDirectoryItem(handle, url, listitem, False)
    
    # 3. Reset denních dat
    listitem = xbmcgui.ListItem("Resetovat denní data")
    listitem.setArt({'thumb': 'DefaultAddonContextItem.png'})
    utils.set_video_info_tag(listitem, {'plot': 'Smaže pouze denní statistiky'})
    url = f"plugin://{addon.getAddonInfo('id')}?action=ai_usage_reset&period=daily"
    xbmcplugin.addDirectoryItem(handle, url, listitem, False)
    
    # 4. Reset podle poskytovatelů
    summary = get_usage_summary()
    providers = summary.get("providers", {})
    
    if any(p.get("total_requests", 0) > 0 for p in providers.values()):
        listitem = xbmcgui.ListItem("─ Reset podle poskytovatelů ─")
        listitem.setArt({'thumb': 'DefaultAddonNone.png'})
        xbmcplugin.addDirectoryItem(handle, "", listitem, False)
        
        for provider, data in providers.items():
            if data.get("total_requests", 0) > 0:
                listitem = xbmcgui.ListItem(f"Resetovat {provider.upper()}")
                listitem.setArt({'thumb': 'DefaultAddonProgram.png'})
                utils.set_video_info_tag(listitem, {'plot': f'Smaže všechna data pro poskytovatele {provider}'})
                url = f"plugin://{addon.getAddonInfo('id')}?action=ai_usage_reset&period=provider:{provider}"
                xbmcplugin.addDirectoryItem(handle, url, listitem, False)
    
    xbmcplugin.endOfDirectory(handle)

def handle_ai_usage_reset(params, addon):
    """Zpracuje reset AI usage dat."""
    period = params.get('period', 'all')
    
    # Mapping pro confirmation dialogs
    period_names = {
        'all': 'všechna data',
        'monthly': 'měsíční data', 
        'daily': 'denní data'
    }
    
    if period.startswith('provider:'):
        provider = period.split(':', 1)[1]
        period_name = f'data pro poskytovatele {provider.upper()}'
    else:
        period_name = period_names.get(period, period)
    
    # Potvrzení
    dialog = xbmcgui.Dialog()
    if not dialog.yesno(
        "Potvrzení resetu",
        f"Opravdu chcete resetovat {period_name}?",
        "Tato akce je nevratná!"
    ):
        return
    
    # Provedení resetu
    success = reset_usage_data(period)
    
    if success:
        dialog.notification(
            "AI Usage Reset",
            f"Úspěšně resetováno: {period_name}",
            xbmcgui.NOTIFICATION_INFO,
            3000
        )
        # Refresh aktuální container
        xbmc.executebuiltin('Container.Refresh')
    else:
        dialog.notification(
            "AI Usage Reset - Chyba",
            f"Chyba při resetování: {period_name}",
            xbmcgui.NOTIFICATION_ERROR,
            5000
        )

def handle_ai_usage_export(addon):
    """Exportuje AI usage statistiky do textového formátu."""
    summary = get_usage_summary()
    
    if "error" in summary:
        dialog = xbmcgui.Dialog()
        dialog.notification(
            "Export Error",
            f"Chyba při načítání dat: {summary['error']}",
            xbmcgui.NOTIFICATION_ERROR,
            5000
        )
        return
    
    # Sestavení export textu
    lines = []
    lines.append("=== WCS AI USAGE STATISTIKY ===")
    lines.append(f"Exportováno: {summary.get('last_updated', 'N/A')}")
    lines.append("")
    
    # Celkové statistiky s token breakdown
    totals = summary.get("totals", {})
    providers = summary.get("providers", {})
    
    # Celkové tokeny
    total_input = sum(sum(m.get("input_tokens", 0) for m in p.get("models", {}).values()) for p in providers.values())
    total_cached = sum(sum(m.get("cached_tokens", 0) for m in p.get("models", {}).values()) for p in providers.values())
    total_output = sum(sum(m.get("output_tokens", 0) for m in p.get("models", {}).values()) for p in providers.values())
    
    lines.append("CELKOVÝ PŘEHLED:")
    lines.append(f"  Celkové náklady: {format_cost(totals.get('cost', 0.0), addon)}")
    lines.append(f"  Celkem požadavků: {totals.get('requests', 0)}")
    lines.append(f"  Průměr na požadavek: {format_cost(totals.get('avg_cost_per_request', 0.0), addon)}")
    lines.append(f"  Input tokeny: {format_tokens(total_input)}")
    if total_cached > 0:
        lines.append(f"  Cached tokeny: {format_tokens(total_cached)}")
    lines.append(f"  Output tokeny: {format_tokens(total_output)}")
    lines.append(f"  Celkem tokenů: {format_tokens(total_input + total_cached + total_output)}")
    lines.append("")
    
    # Breakdown podle poskytovatelů s tokeny
    lines.append("BREAKDOWN PODLE POSKYTOVATELŮ:")
    for provider, data in providers.items():
        if data.get("total_requests", 0) > 0:
            # Provider token breakdown
            provider_input = sum(m.get("input_tokens", 0) for m in data.get("models", {}).values())
            provider_cached = sum(m.get("cached_tokens", 0) for m in data.get("models", {}).values())
            provider_output = sum(m.get("output_tokens", 0) for m in data.get("models", {}).values())
            
            lines.append(f"  {provider.upper()}:")
            lines.append(f"    Náklady: {format_cost(data.get('total_cost', 0.0), addon)}")
            lines.append(f"    Požadavky: {data.get('total_requests', 0)}")
            lines.append(f"    Input tokeny: {format_tokens(provider_input)}")
            if provider_cached > 0:
                lines.append(f"    Cached tokeny: {format_tokens(provider_cached)}")
            lines.append(f"    Output tokeny: {format_tokens(provider_output)}")
            
            # Modely s token detaily
            models = data.get("models", {})
            if models:
                lines.append("    Modely:")
                sorted_models = sorted(models.items(), key=lambda x: x[1].get("total_cost", 0.0), reverse=True)
                for model_name, model_data in sorted_models:
                    input_t = model_data.get('input_tokens', 0)
                    cached_t = model_data.get('cached_tokens', 0)
                    output_t = model_data.get('output_tokens', 0)
                    
                    token_detail = f"in:{format_tokens(input_t)}"
                    if cached_t > 0:
                        token_detail += f" cached:{format_tokens(cached_t)}"
                    token_detail += f" out:{format_tokens(output_t)}"
                    
                    lines.append(f"      {model_name}: {format_cost(model_data.get('total_cost', 0.0), addon)} ({model_data.get('requests', 0)}x) - {token_detail}")
            lines.append("")
    
    # Top modely s token detaily
    top_models = summary.get("top_models", [])
    if top_models:
        lines.append("TOP 10 MODELŮ PODLE NÁKLADŮ:")
        for i, model_info in enumerate(top_models[:10], 1):
            input_t = model_info.get('input_tokens', 0)
            cached_t = model_info.get('cached_tokens', 0)
            output_t = model_info.get('output_tokens', 0)
            
            token_detail = f"in:{format_tokens(input_t)}"
            if cached_t > 0:
                token_detail += f" cached:{format_tokens(cached_t)}"
            token_detail += f" out:{format_tokens(output_t)}"
            
            lines.append(
                f"  {i}. {model_info['provider']}/{model_info['model']} - "
                f"{format_cost(model_info['cost'], addon)} ({model_info['requests']}x) - {token_detail}"
            )
        lines.append("")
    
    # Zobrazení v textovém dialogu
    export_text = '\n'.join(lines)
    dialog = xbmcgui.Dialog()
    dialog.textviewer("AI Usage Statistiky - Export", export_text) 