# wcs/webshare_login_dialog.py

import xbmcgui
import xbmcaddon
import xbmc
from wcs.ui.dialogs.guide.DialogWebshareGuide import WebshareGuideDialog


"""Webshare login dialog a otevření jednotného guide dialogu."""


class WebshareLoginDialog(xbmcgui.WindowXMLDialog):
    """Dialog pro přihlášení k Webshare.cz s možností zadání údajů nebo zobrazení návodu."""
    
    def __init__(self, *args, **kwargs):
        self.on_success_callback = kwargs.pop('on_success_callback', None)
        super().__init__(*args, **kwargs)
        self.addon = xbmcaddon.Addon()
        self.credentials_entered = False
        self.dialog_closed = False
        
    def onInit(self):
        """Inicializace dialogu."""
        xbmc.log("[plugin.video.milionar] WebshareLoginDialog: Dialog initialized", level=xbmc.LOGINFO)
        
    def onClick(self, controlId):
        """Obsluha kliknutí na tlačítka."""
        if controlId == 7001:  # Zadat údaje
            xbmc.log("[plugin.video.milionar] WebshareLoginDialog: Enter credentials clicked", level=xbmc.LOGINFO)
            self._handle_enter_credentials()
            
        elif controlId == 7002:  # Návod
            xbmc.log("[plugin.video.milionar] WebshareLoginDialog: Guide clicked", level=xbmc.LOGINFO)
            self._handle_show_guide()
            
        elif controlId == 7003:  # Zavřít
            xbmc.log("[plugin.video.milionar] WebshareLoginDialog: Close clicked", level=xbmc.LOGINFO)
            self.dialog_closed = True
            self.close()
            
    def _handle_enter_credentials(self):
        """Zpracuje zadávání přihlašovacích údajů a ihned ověří jejich platnost."""
        from wcs.webshare import WebshareClient as webshare_api
        # Krok 1: Zadání uživatelského jména
        keyboard = xbmc.Keyboard('', 'Zadejte uživatelské jméno pro Webshare.cz')
        keyboard.doModal()
        if not keyboard.isConfirmed():
            return  # Uživatel zrušil
        username = keyboard.getText().strip()
        if not username:
            xbmcgui.Dialog().ok("Chyba zadání", "Uživatelské jméno nemůže být prázdné.")
            return
        # Krok 2: Zadání hesla
        keyboard = xbmc.Keyboard('', 'Zadejte heslo pro Webshare.cz', True)  # True = skrýt znaky
        keyboard.doModal()
        if not keyboard.isConfirmed():
            return  # Uživatel zrušil
        password = keyboard.getText().strip()
        if not password:
            xbmcgui.Dialog().ok("Chyba zadání", "Heslo nemůže být prázdné.")
            return
        # Ověření platnosti údajů přes Webshare API
        token = webshare_api.login(username, password, self.addon, lambda *args, **kwargs: None, open_settings_on_fail=False)
        if token:
            # Údaje jsou platné, uložíme je
            xbmc.log(f"[plugin.video.milionar] WebshareLoginDialog: Saving credentials - username: {username}, password: {'*' * len(password)}", level=xbmc.LOGINFO)
            self.addon.setSetting('wsuser', username)
            self.addon.setSetting('wspass', password)
            self.addon.setSetting('webshare_status', f'Přihlášen jako: {username}')
            self.addon.setSetting('token', token)
            xbmc.log(f"[plugin.video.milionar] WebshareLoginDialog: Credentials saved successfully", level=xbmc.LOGINFO)
            # Po přihlášení: jednotná logika načtení VIP + notifikace
            try:
                # Složit notifikaci: "{Logged in:} {username} – {VIP text}"
                prefix = f"{self.addon.getLocalizedString(32178)} {username}, "
                webshare_api.refresh_vip_info(self.addon, token_override=token, show_notification=True, log_context="login_dialog", notification_prefix=prefix)
            except Exception as e:
                xbmc.log(f"[plugin.video.milionar] WebshareLoginDialog VIP info error: {e}", level=xbmc.LOGWARNING)
            self.credentials_entered = True
            if self.on_success_callback:
                xbmc.log("[plugin.video.milionar] WebshareLoginDialog: Calling on_success_callback", level=xbmc.LOGINFO)
                self.on_success_callback()
            self.close()

        else:
            # Údaje jsou neplatné, nabídneme zadání znovu nebo zavření
            self.addon.setSetting('webshare_status', 'Nepřihlášen')
            self.addon.setSetting('token', '')
            confirm = xbmcgui.Dialog().yesno(
                "Neplatné přihlašovací údaje",
                "Zadané údaje k Webshare.cz jsou neplatné. Zkontrolujte prosím, že jste je zadali správně.",
                yeslabel="Zadat znovu",
                nolabel="Zavřít"
            )
            if confirm:
                self._handle_enter_credentials()
                return
            else:
                return
            
    def _handle_show_guide(self):
        """Zobrazí návod pro registraci na Webshare.cz."""
        try:
            # Otevření jednotného guide dialogu
            guide_dialog = WebshareGuideDialog(
                'guide/DialogWebshareGuide.xml',
                self.addon.getAddonInfo('path')
            )
            guide_dialog.doModal()
            del guide_dialog
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] WebshareLoginDialog: Error showing guide: {e}", level=xbmc.LOGERROR)
            xbmcgui.Dialog().ok("Chyba", f"Došlo k chybě při zobrazení návodu: {e}")
            
    def get_result(self):
        """Vrátí výsledek dialogu."""
        return {
            'credentials_entered': self.credentials_entered,
            'dialog_closed': self.dialog_closed
        }


def show_webshare_login_dialog(addon, on_success_callback=None):
    """Zobrazí dialog pro přihlášení k Webshare.cz. Pokud je předán callback, zavolá jej po úspěšném přihlášení."""
    try:
        dialog = WebshareLoginDialog('login/DialogWebshareLogin.xml', addon.getAddonInfo('path'), on_success_callback=on_success_callback)
        dialog.doModal()
        # Callback je nyní volán zevnitř dialogu, takže zde ho již nevoláme.
        return dialog.get_result()
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] show_webshare_login_dialog: Error: {e}", level=xbmc.LOGERROR)
        return {'credentials_entered': False, 'dialog_closed': True} 