import xbmc
import xbmcaddon
import xbmcgui
from wcs.ai import AIHandler as ai_handler
from wcs import user_data
from wcs.ai.AIProvider import ai_provider


class AdventureGameController:
    """
    Kontrolér specificky pro textové adventure hry.
    Obsahuje logiku pro interaktivní příběhy inspirované filmy.
    """
    
    def __init__(self, title="", plot="", genre="", media_type="movie", episode_title="", season_number=None, episode_number=None, actor_data=None):
        self.addon = xbmcaddon.Addon()
        
        # Informace o médiu
        self.title = title
        self.plot = plot
        self.genre = genre
        self.media_type = media_type  # "movie" nebo "series"
        self.episode_title = episode_title  # Pro seriály
        self.season_number = season_number  # Pro seriály
        self.episode_number = episode_number  # Pro seriály
        
        # Data o herci (pokud media_type == 'actor')
        self.actor_data = actor_data or {}
        
        # Základní herní stav
        self.game_state = "intro"
        self.game_type = "adventure"
        
        # UI callbacks
        self.update_display_callback = None
        self.update_buttons_callback = None
        self.update_sidebar_callback = None
        self.show_error_callback = None
        self.set_focus_callback = None
        self.start_countdown_callback = None
        self.close_dialog_callback = None
        
        # Adventure-specifické vlastnosti
        self.story_history = []
        self.choices_history = []
        self.current_choices = []
        self.turn_number = 0
    
    def set_ui_callbacks(self, update_display, update_buttons, update_sidebar, show_error, set_focus=None, start_countdown=None, close_dialog=None):
        """Nastaví callback funkce pro komunikaci s UI."""
        self.update_display_callback = update_display
        self.update_buttons_callback = update_buttons
        self.update_sidebar_callback = update_sidebar
        self.show_error_callback = show_error
        self.set_focus_callback = set_focus
        self.start_countdown_callback = start_countdown
        self.close_dialog_callback = close_dialog

    def set_return_to_menu_callback(self, callback):
        """Nastaví callback pro návrat na hlavní menu."""
        self.return_to_menu_callback = callback

    def _add_ai_info_header(self, text: str) -> str:
        """Přidá informaci o AI modelu na začátek textu velmi malým písmem."""
        # AI info odstraněno na požádání uživatele
        return text
        
    def _track_ai_usage(self, usage, usage_type):
        """Společná metoda pro sledování AI usage."""
        if usage:
            from wcs.ai.UsageTracker import track_ai_usage
            provider = ai_provider.get_current_provider()
            model = ai_provider.get_current_model()
            track_ai_usage(provider, model, usage, usage_type)
        
    def start_specific_game(self):
        """Spustí textovou adventuru s welcome screenem."""
        self.game_state = "adventure_welcome"
        self.story_history = []
        self.choices_history = []
        self.turn_number = 0
        
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
            
        # Zobrazit welcome screen
        self.show_adventure_welcome()

    def show_adventure_welcome(self):
        """Zobrazí uvítací obrazovku textové adventury."""
        # Sestavení textu podle typu média
        if self.media_type == "series" and self.episode_title:
            media_info = f'"{self.title}" - {self.episode_title}'
            if self.season_number and self.episode_number:
                media_info += f" (S{self.season_number:02d}E{self.episode_number:02d})"
        else:
            media_info = f'"{self.title}"'
        
        # Přidání žánru pokud je dostupný
        genre_text = f" • Žánr: {self.genre}" if self.genre else ""
        
        # Krátký popis děje pokud je dostupný
        plot_preview = ""
        if self.plot and len(self.plot) > 50:
            plot_preview = f"\nDĚJ: {self.plot[:150]}{'...' if len(self.plot) > 150 else ''}\n"
        
        # Rozlišení typu média pro popis
        if self.media_type == "series":
            media_source = f"ze seriálu \"{self.title}\""
        elif self.media_type == "actor":
            media_source = f"hercem \"{self.title}\""
        elif self.media_type == "director":
            media_source = f"režisérem \"{self.title}\""
        elif self.media_type == "writer":
            media_source = f"scénáristou \"{self.title}\""
        elif self.media_type == "creator":
            media_source = f"tvůrcem \"{self.title}\""
        else:
            media_source = f"z filmu \"{self.title}\""
        
        welcome_text = f"""CO VÁS ČEKÁ:
• Interaktivní příběh {media_source}
• 5 kol strategického rozhodování
• Vaše volby ovlivní průběh a konec
• Hrajete jako hlavní postava ze světa příběhu
{plot_preview}
Připravte se vstoupit do světa dobrodružství!

Jste připraveni začít svou cestu?"""

        # Nastavit property pro welcome sidebar
        self._set_adventure_film_info()

        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(welcome_text))
        if self.update_buttons_callback:
            self.update_buttons_callback("Začít příběh", "Pravidla", "Zpět na menu", "Zavřít")
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
        # Nastavit fokus na tlačítko "Začít příběh" (index 0)
        if self.set_focus_callback:
            self.set_focus_callback(0)

    def _set_adventure_film_info(self):
        """Nastaví property s informacemi o filmu pro welcome sidebar."""
        try:
            # Získat reference na hlavní window pro nastavení property
            import xbmcgui
            window = xbmcgui.Window(xbmcgui.getCurrentWindowId())
            
            # Sestavit informace o filmu podle typu média
            if self.media_type == "series" and self.episode_title:
                film_info = f"Seriál: {self.title}\nEpizoda: {self.episode_title}"
                if self.season_number and self.episode_number:
                    film_info += f" (S{self.season_number:02d}E{self.episode_number:02d})"
            elif self.media_type == "series":
                film_info = f"Seriál: {self.title}"
            elif self.media_type == "actor":
                film_info = f"Herec: {self.title}"
            elif self.media_type == "director":
                film_info = f"Režisér: {self.title}"
            elif self.media_type == "writer":
                film_info = f"Scénárista: {self.title}"
            elif self.media_type == "creator":
                film_info = f"Tvůrce: {self.title}"
            else:
                film_info = f"Film: {self.title}"
            
            # Přidat žánr pokud je dostupný
            if self.genre:
                film_info += f"\nŽánr: {self.genre}"
            
            window.setProperty('WCS.Adventure.FilmInfo', film_info)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error setting adventure film info: {e}", level=xbmc.LOGERROR)

    def show_adventure_instructions(self):
        """Zobrazí návod k textové adventuře."""
        # Rozlišení typu média pro návod
        if self.media_type == "series":
            media_reference = f"seriálem \"{self.title}\""
            media_basis = "seriálu"
        elif self.media_type == "actor":
            media_reference = f"hercem \"{self.title}\""
            media_basis = "herci"
        elif self.media_type == "director":
            media_reference = f"režisérem \"{self.title}\""
            media_basis = "režiséru"
        elif self.media_type == "writer":
            media_reference = f"scénáristou \"{self.title}\""
            media_basis = "scénáristovi"
        elif self.media_type == "creator":
            media_reference = f"tvůrcem \"{self.title}\""
            media_basis = "tvůrci"
        else:
            media_reference = f"filmem \"{self.title}\""
            media_basis = "filmu"
        
        instructions_text = f"""PRŮBĚH HRY:
• Prožijete příběh inspirovaný {media_reference}
• Celkem 5 kol rozhodování
• Každé kolo máte 3 možnosti na výběr
• Vaše volby ovlivní:
  - Průběh příběhu
  - Reakce postav
  - Finální konec

HERNÍ MECHANIKA:
• AI vytvoří situace na základě {media_basis}
• Vždy máte 3 možnosti volby (A, B, C)
• Příběh se vyvíjí podle vašich rozhodnutí
• Každá hra je unikátní

Jste připraveni na dobrodružství?"""

        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(instructions_text))
        if self.update_buttons_callback:
            self.update_buttons_callback("Začít příběh", "Zpět", "Zpět na menu", "Zavřít")

    def load_adventure_game(self):
        """Načte začátek textové adventury."""
        self.game_state = "adventure_playing"
        self.turn_number = 1
        
        # Zobrazit loading
        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(f"AI připravuje textovou adventuru...\n\nTéma: {self.title}\n\nGeneruji interaktivní příběh s možnostmi voleb...\n\nChvíli strpení..."))
        if self.update_buttons_callback:
            self.update_buttons_callback("Načítání...", "...", "...", "...")
        if self.update_sidebar_callback:
            self.update_sidebar_callback()
        
        try:
            # Volání AI pro úvodní text
            if self.media_type == 'actor':
                intro_text, choices, usage = ai_handler.start_actor_text_game(
                    actor_name=self.title,
                    actor_bio=self.plot,
                    known_for=[item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data else []
                )
            elif self.media_type == 'director':
                intro_text, choices, usage = ai_handler.start_director_text_game(
                    director_name=self.title,
                    director_bio=self.plot,
                    known_for=[item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data else []
                )
            elif self.media_type == 'writer':
                intro_text, choices, usage = ai_handler.start_writer_text_game(
                    writer_name=self.title,
                    writer_bio=self.plot,
                    known_for=[item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data else []
                )
            elif self.media_type == 'creator':
                intro_text, choices, usage = ai_handler.start_creator_text_game(
                    creator_name=self.title,
                    creator_bio=self.plot,
                    known_for=[item.get('title') or item.get('name') for item in self.actor_data.get('known_for', [])] if self.actor_data else []
                )
            else:
                intro_text, choices, usage = ai_handler.start_text_game(
                    movie_title=self.title,
                    movie_plot=self.plot,
                    movie_genre=self.genre
                )
            
            # Uložení usage statistik
            self._track_ai_usage(usage, "game")
            
            # Uložení do historie
            self.story_history.append(intro_text)
            self.current_choices = choices
            
            # Aktualizace UI
            if self.update_display_callback:
                self.update_display_callback(self._add_ai_info_header(self._get_current_story_text()))
            if self.update_buttons_callback:
                self.update_buttons_callback(
                    f"{choices[0]}" if len(choices) > 0 else "",
                    f"{choices[1]}" if len(choices) > 1 else "",
                    f"{choices[2]}" if len(choices) > 2 else "",
                    "Zavřít"
                )
            if self.update_sidebar_callback:
                self.update_sidebar_callback()
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error starting adventure: {e}", level=xbmc.LOGERROR)
            if self.show_error_callback:
                self.show_error_callback("Chyba při spouštění textové hry. Zkuste to znovu.")

    def make_adventure_choice(self, choice_index):
        """Zpracuje volbu v textové hře."""
        if choice_index >= len(self.current_choices):
            return
            
        choice_made = self.current_choices[choice_index]
        self.choices_history.append(choice_made)
        
        # IHNED přidat uživatelskou volbu do historie a aktualizovat UI
        user_choice_text = f"HRÁČ: {choice_made}"
        self.story_history.append(user_choice_text)
        
        # Okamžitě aktualizovat zobrazení s uživatelskou volbou
        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(self._get_current_story_text()))
        
        self.turn_number += 1
        
        # Zobrazit loading pro AI odpověď
        if self.update_buttons_callback:
            self.update_buttons_callback("AI přemýšlí...", "...", "...", "...")
        
        try:
            if self.media_type == 'actor':
                new_text, next_choices, is_ending, usage = ai_handler.continue_actor_text_game(
                    actor_name=self.title,
                    story_history=self.story_history,
                    choice_made=choice_made,
                    turn_number=self.turn_number
                )
            elif self.media_type == 'director':
                new_text, next_choices, is_ending, usage = ai_handler.continue_director_text_game(
                    director_name=self.title,
                    story_history=self.story_history,
                    choice_made=choice_made,
                    turn_number=self.turn_number
                )
            elif self.media_type == 'writer':
                new_text, next_choices, is_ending, usage = ai_handler.continue_writer_text_game(
                    writer_name=self.title,
                    story_history=self.story_history,
                    choice_made=choice_made,
                    turn_number=self.turn_number
                )
            elif self.media_type == 'creator':
                new_text, next_choices, is_ending, usage = ai_handler.continue_creator_text_game(
                    creator_name=self.title,
                    story_history=self.story_history,
                    choice_made=choice_made,
                    turn_number=self.turn_number
                )
            else:
                new_text, next_choices, is_ending, usage = ai_handler.continue_text_game(
                    movie_title=self.title,
                    story_history=self.story_history,
                    choice_made=choice_made,
                    turn_number=self.turn_number
                )
            
            # Uložení usage statistik
            self._track_ai_usage(usage, "game")
            
            # Přidat AI odpověď do historie
            self.story_history.append(new_text)
            self.current_choices = next_choices
            self.is_ending = is_ending
            
            # Aktualizovat zobrazení s AI odpovědí
            if self.update_display_callback:
                self.update_display_callback(self._add_ai_info_header(self._get_current_story_text()))
            if self.update_buttons_callback:
                if is_ending:
                    self.update_buttons_callback("", "", "", "")
                else:
                    self.update_buttons_callback(
                        f"{next_choices[0]}" if len(next_choices) > 0 else "",
                        f"{next_choices[1]}" if len(next_choices) > 1 else "",
                        f"{next_choices[2]}" if len(next_choices) > 2 else "",
                        "Zavřít"
                    )
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error making choice: {e}", level=xbmc.LOGERROR)
            self.current_choices = []
            if self.update_display_callback:
                self.update_display_callback(self._add_ai_info_header(f"Chyba při získávání další části příběhu: {e}"))
            if self.update_buttons_callback:
                self.update_buttons_callback("", "", "", "")

    def end_adventure(self):
        """Ukončí textovou adventuru."""
        self.game_state = "ended"
        
        final_story = self._get_current_story_text()
        final_story += f"\n\n" + "═" * 50 + "\n"
        final_story += "KONEC PŘÍBĚHU\n\n"
        final_story += f"Dokončili jste příběh ve {min(self.turn_number - 1, 5)} kolech.\n"
        final_story += f"Celkem jste učinili {len(self.choices_history)} voleb.\n\n"
        final_story += "Děkujeme za hraní!"
        
        if self.update_display_callback:
            self.update_display_callback(self._add_ai_info_header(final_story))
        if self.update_buttons_callback:
            self.update_buttons_callback("Hrát znovu", "Zpět na menu", "Zavřít", "")
        if self.update_sidebar_callback:
            self.update_sidebar_callback()

    def make_choice(self, choice_index):
        """Hlavní metoda pro zpracování voleb."""
        self.handle_game_choice(choice_index)
    
    def handle_game_choice(self, choice_index):
        """Zpracuje volbu specifickou pro adventure hru."""
        if self.game_state == "adventure_welcome":
            if choice_index == 0:  # Začít příběh
                self.load_adventure_game()
            elif choice_index == 1:  # Návod
                self.game_state = "adventure_instructions"
                self.show_adventure_instructions()
            elif choice_index == 2:  # Zpět na menu
                self.start_game_selection()
            elif choice_index == 3:  # Zavřít
                self.end_game()
        elif self.game_state == "adventure_instructions":
            if choice_index == 0:  # Začít příběh
                self.load_adventure_game()
            elif choice_index == 1:  # Zpět
                self.game_state = "adventure_welcome"
                self.show_adventure_welcome()
            elif choice_index == 2:  # Zpět na menu
                self.start_game_selection()
            elif choice_index == 3:  # Zavřít
                self.end_game()
        elif self.game_state == "adventure_playing":
            if choice_index == 3:  # Zavřít tlačítko během hry
                self.end_game()
            else:
                self.make_adventure_choice(choice_index)
        elif self.game_state == "ended":
            if choice_index == 0:  # Hrát znovu
                self.start_specific_game()
            elif choice_index == 1:  # Zpět na menu
                self.start_game_selection()
            elif choice_index == 2:  # Zavřít
                self.end_game()

    def get_game_specific_sidebar_items(self):
        """Vrací sidebar položky specifické pro adventure hru."""
        if self.game_state == "adventure_welcome":
            return ["Adventure výzva", f"Film: {self.title[:20]}", "5 kol čeká"]
        elif self.game_state == "adventure_instructions":
            return ["Návod k adventuře", "Pravidla hry", f"Film: {self.title[:15]}"]
        elif self.game_state == "adventure_playing":
            progress = f"{min(self.turn_number, 5)}/5"
            items = [
                f"Kolo: {progress}",
                f"Voleb: {len(self.choices_history)}",
                "Postava vybrána" if self.turn_number > 1 else "Výběr postavy"
            ]
            
            # Přidáme progress bar
            if self.turn_number <= 5:
                filled = "█" * min(self.turn_number, 5)
                empty = "░" * (5 - min(self.turn_number, 5))
                items.append(f"Progress: {filled}{empty}")
            return items
        elif self.game_state == "ended":
            final_turns = min(self.turn_number - 1, 5)
            return [
                "Hra skončena",
                f"Dokončeno: {final_turns}/5",
                f"Celkem voleb: {len(self.choices_history)}"
            ]
        else:
            return super().get_sidebar_items()

    def get_sidebar_items(self):
        """Vrací aktuální sidebar položky."""
        return self.get_game_specific_sidebar_items()

    def _get_current_story_text(self):
        """Získá aktuální text příběhu."""
        # Vrací text příběhu pro zobrazení v UI, nejnovější dole
        return "\n\n".join(self.story_history)

    def end_game(self):
        """Ukončí hru - zavře dialog."""
        if self.close_dialog_callback:
            self.close_dialog_callback()

    def start_game_selection(self):
        """Návrat na hlavní výběr typu hry."""
        if hasattr(self, 'return_to_menu_callback') and self.return_to_menu_callback:
            self.return_to_menu_callback()
        else:
            # Fallback pro zpětnou kompatibilitu
            if self.close_dialog_callback:
                self.close_dialog_callback() 