# wcs/search_dialog.py
import xbmc
import xbmcgui
import xbmcaddon
import xbmcplugin
import requests
import xml.etree.ElementTree as ET
from urllib.parse import quote_plus, unquote
import re

from wcs.webshare import WebshareClient as webshare_api
from wcs import utils
from wcs.utils import (
    extract_clean_title, 
    extract_quality, 
    extract_language, 
    extract_format, 
    format_size, 
    calculate_rating,
    is_valid_file_size  # Přesunuto z search_dialog.py
)
from wcs.search.MovieSearch import sort_files_by_priority
from wcs.search.SeriesSearch import sort_series_files_by_priority  # Jen jednou na začátku!
from wcs.playback.DialogMediaInfoWithGame import get_autoplay_manager
from wcs.ui import UI as ui  # Import pro popinfo funkci

def format_quality_with_color(quality):
    """Formátuje kvalitu s barevným kódováním pro Kodi."""
    if not quality:
        return ""
    
    # Barevné schéma pro kvality
    color_map = {
        '4K': 'FFFF4444',      # Červená pro 4K
        'REMUX': 'FFFF00FF',   # Magenta pro REMUX
        '1080p': 'FF44AAFF',   # Modrá pro 1080p
        '720p': 'FFFF9944',    # Oranžová pro 720p
        '480p': 'FFDDDD44',    # Žlutá pro 480p
        '360p': 'FFAA8844',    # Hnědá pro 360p
        'SD': 'FF999999'       # Šedá pro SD
    }
    
    # Najdi barvu pro kvalitu
    color = color_map.get(quality, 'FFFFFFFF')  # Bílá jako fallback
    
    # Vrať kvalitu s tečkou a barevným formátováním
    return f"• [COLOR {color}]{quality}[/COLOR]"

# is_valid_file_size je nyní v utils.py jako sdílená funkce

class SearchDialog(xbmcgui.WindowXMLDialog):
    """
    Dialogové okno pro zobrazení výsledků vyhledávání z Webshare.
    Používá search/DialogWcsSearch.xml.
    """
    def __init__(self, xml_filename, addon_path, *args, **kwargs):
        self.xml_filename = xml_filename # Uložíme si jméno XML pro pozdější logiku
        super().__init__(xml_filename, addon_path, "Default", "1080i")
        # Uložíme si naše kwargs
        self.search_query = kwargs.get('search_query', '')
        self.movie_title = kwargs.get('movie_title', '')
        self.movie_year = kwargs.get('movie_year', '')
        self.poster_url = kwargs.get('poster_url', '')  # Přidáno pro poster
        self.rating = kwargs.get('rating', '')         # Hodnocení filmu
        self.genre = kwargs.get('genre', '')           # Žánr filmu
        self.director = kwargs.get('director', '')     # Režisér
        self.runtime = kwargs.get('runtime', '')       # Délka filmu
        self.plot = kwargs.get('plot', '')            # Přidaný plot pro film
        self.addon = xbmcaddon.Addon()
        # Možnost předat už připravené výsledky
        self.search_results = kwargs.get('search_results', [])
        self.selected_result = None
        
        # Autoplay podpora
        self.is_episode = kwargs.get('is_episode', False)
        self.series_name = kwargs.get('series_name', '')
        self.season_number = kwargs.get('season_number', None)
        self.episode_number = kwargs.get('episode_number', None)
        self.tmdb_id = kwargs.get('tmdb_id', None)
        self.autoplay_manager = get_autoplay_manager()
        
        # Vlastní stav autoplay přepínače - načteme z globálního nastavení
        # Ale pouze pro epizody a pokud je to povoleno globálně
        # POZN: Autoplay funguje POUZE pro seriály z knihovny (natvrdo nastaveno)
        self.autoplay_enabled = (
            self.is_episode and 
            self.addon.getSettingBool('autoplay_enabled_global') and
            True  # Autoplay POUZE pro knihovnu - natvrdo nastaveno
        )
        
        # Nový parametr
        self.preferred_quality = kwargs.get('preferred_quality', None)
        self.from_library = kwargs.get('from_library', False)
        self.fanart_url = kwargs.get('fanart_url', '') # Přidáno pro fanart
        self.episode_title = kwargs.get('episode_title', '') # Přidáno pro název epizody
        self.episode_thumb = kwargs.get('episode_thumb', '') # Still epizody pro úvodní dialog
        
        
        # Nový parametr pro manuální vyhledávání
        self.manual_search = kwargs.get('manual_search', False)
        
    def onInit(self):
        """
        Volá se při inicializaci okna. Zde načteme výsledky vyhledávání a naplníme seznam.
        """
        xbmc.log(f"[plugin.video.milionar] Initializing SearchDialog for query: {self.search_query}", level=xbmc.LOGINFO)
        if not self.search_query:
            xbmc.log("[plugin.video.milionar] Error: Empty search query for SearchDialog.", level=xbmc.LOGERROR)
            self.close()
            return

        # Nastavíme texty v dialogu
        dialog_title = f"Hledaný výraz: {self.search_query}"
        self.setProperty('WCS.Search.Title', dialog_title)

        info_text = f"Hledaný výraz: {self.search_query}"
        self.setProperty('WCS.Search.Info', info_text)

        # Nastavíme poster do dialogu – vždy poster seriálu (UI), ne still epizody
        if hasattr(self, 'poster_url') and self.poster_url:
            self.setProperty('WCS.Search.Poster', self.poster_url)
        else:
            self.setProperty('WCS.Search.Poster', '')

        # Nastavíme název filmu/seriálu (MovieTitle)
        if hasattr(self, 'movie_title') and self.movie_title:
            self.setProperty('WCS.Search.MovieTitle', self.movie_title)
        else:
            self.setProperty('WCS.Search.MovieTitle', '')

        # Nastavíme fanart, pokud je k dispozici (předpokládáme, že je v self.fanart_url)
        fanart_url = getattr(self, 'fanart_url', '')
        if fanart_url:
            self.setProperty('WCS.Search.Fanart', fanart_url)
        else:
            self.setProperty('WCS.Search.Fanart', '')

        # Nastavíme metadata filmu
        if hasattr(self, 'rating') and self.rating:
            try:
                rating_val = float(self.rating)
                rating_str = f"{rating_val:.1f}/10"
            except Exception:
                rating_str = str(self.rating)
            self.setProperty('WCS.Search.Rating', rating_str)
        else:
            self.setProperty('WCS.Search.Rating', '')

        if hasattr(self, 'genre') and self.genre:
            self.setProperty('WCS.Search.Genre', self.genre)
        else:
            self.setProperty('WCS.Search.Genre', '')

        if hasattr(self, 'director') and self.director:
            self.setProperty('WCS.Search.Director', self.director)
        else:
            self.setProperty('WCS.Search.Director', '')

        if hasattr(self, 'runtime') and self.runtime:
            self.setProperty('WCS.Search.Runtime', self.runtime)
        else:
            self.setProperty('WCS.Search.Runtime', '')

        # Nastavíme rok, pokud byl předán
        if hasattr(self, 'movie_year') and self.movie_year:
            self.setProperty('WCS.Search.Year', self.movie_year)
        else:
            self.setProperty('WCS.Search.Year', '')

        # Nastavíme plot (popis), pokud byl předán 
        if hasattr(self, 'plot') and self.plot:
            self.setProperty('WCS.Search.Plot', self.plot)
        
        # Specifika pro manuální vyhledávání
        if self.manual_search:
            # V manuálním režimu chceme jako titulek vidět jen hledaný výraz a informaci o zdroji
            self.setProperty('WCS.Search.MovieTitle', f"Webshare: {self.search_query}")
            self.setProperty('WCS.Search.Info', f"Hledaný výraz: {self.search_query}")
            
            # Nastavíme ikonu doplňku jako poster
            addon_icon = 'special://home/addons/plugin.video.milionar/resources/media/icons/Milionar.png'
            self.setProperty('WCS.Search.Poster', addon_icon)
            
            # Nastavíme popis na informaci o přímém vyhledávání
            self.setProperty('WCS.Search.Plot', f"Zobrazeny jsou výsledky přímého vyhledávání souborů na serveru Webshare.cz pro výraz '{self.search_query}'.\n\nTyto výsledky neobsahují metadata z filmových databází (TMDb).")
            
            # Skryjeme ostatní metadata, která by mohla být matoucí
            self.setProperty('WCS.Search.Rating', '')
            self.setProperty('WCS.Search.Genre', 'Toto je přímé vyhledávání souborů na Webshare.cz')
            self.setProperty('WCS.Search.Director', '')
            self.setProperty('WCS.Search.Runtime', '')
            self.setProperty('WCS.Search.Year', '')
        
        # Autoplay nastavení pro epizody
        if self.is_episode:
            self.setProperty('WCS.Episode.IsEpisode', 'true')
            self.setProperty('WCS.Episode.FromLibrary', 'true' if self.from_library else 'false')
            # Nastavení výchozího stavu a labelu
            self._update_autoplay_label()
            xbmc.log(f"[plugin.video.milionar] Episode detected: {self.series_name} S{self.season_number}E{self.episode_number}", level=xbmc.LOGINFO)
        else:
            self.setProperty('WCS.Episode.IsEpisode', 'false')
            self.setProperty('WCS.Episode.FromLibrary', 'false')
            # Nastavení informačního textu pro filmy
            self._update_autoplay_label()
        
        # Logování metadat z knihovny při zobrazení dialogu pro epizodu
        if self.is_episode:
            xbmc.log(
                f"[plugin.video.milionar] Library metadata: series='{self.series_name}', season={self.season_number}, episode={self.episode_number}, tmdb_id={self.tmdb_id}, movie_title='{self.movie_title}', movie_year='{self.movie_year}'",
                level=xbmc.LOGINFO
            )

        # Nastavíme název epizody jako property, pokud je k dispozici
        if hasattr(self, 'episode_title') and self.episode_title:
            self.setProperty('WCS.Search.EpisodeTitle', self.episode_title)
        else:
            self.setProperty('WCS.Search.EpisodeTitle', '')

        # Zobrazíme indikátor načítání
        xbmc.executebuiltin("ActivateWindow(busydialognocancel)")

        # Získáme výsledky z Webshare - jen pokud nejsou už předané
        if not self.search_results:
            self.search_results = self._perform_search()
        else:
            # Seřazení podle typu obsahu - filmy vs seriály
            if self.is_episode:
                # Pro seriály/epizody použijeme search_series.py
                self.search_results = sort_series_files_by_priority(self.search_results, self.search_query)
            elif self.manual_search:
                # Pro manuální vyhledávání použijeme sort_files_by_priority s filter_series=False
                # Chceme vidět všechna videa seřazená podle kvality/relevance
                self.search_results = sort_files_by_priority(self.search_results, self.search_query, filter_series=False)
            else:
                # Pro filmy použijeme search_movies.py (s filtrováním seriálů)
                self.search_results = sort_files_by_priority(self.search_results, self.search_query, filter_series=True)

        # Skryjeme indikátor načítání
        xbmc.executebuiltin("Dialog.Close(busydialognocancel)")

        if not self.search_results:
            # Zobrazit chybu a zavřít okno, pokud se výsledky nepodařilo načíst
            xbmcgui.Dialog().ok("Žádné výsledky", "Nepodařilo se najít žádné odpovídající soubory na Webshare.")
            self.close()
            return

        # Naplníme seznam výsledků
        self._populate_results_list()

        # Nastavíme fokus na tlačítko "Přehrát nejlepší" (klasika) nebo Seznam (AI Sidebar)
        xbmc.sleep(100)  # Krátká pauza, aby se UI stihlo aktualizovat
        
        # FIX: Pokud jde o AI Sidebar, chceme fokus ihned na seznamu (50) pro plynulou navigaci
        if hasattr(self, 'xml_filename') and self.xml_filename and "DialogAIChatSearch.xml" in self.xml_filename:
             self.setFocusId(50)
        else:
             self.setFocusId(100)  # Klasické chování - Play Best

    def _perform_search(self):
        """Provede vyhledávání na Webshare API a vrátí výsledky."""
        try:
            # Revalidace tokenu
            token = webshare_api.revalidate(self.addon, lambda msg, title=None, icon=None: xbmcgui.Dialog().notification(title, msg, icon=icon))
            if not token:
                return []

            # Lepší logování pro diagnostiku
            
            # Volání API
            response = webshare_api.api_call('search', {
                'what': self.search_query,
                'category': 'video',  # Vždy hledáme jen videa
                'sort': '',           # Výchozí řazení
                'limit': 50,          # Více výsledků pro lepší výběr
                'offset': 0,
                'wst': token
            })
            
            # Kontrola odpovědi
            if not response or not response.content:
                xbmc.log("[plugin.video.milionar] Empty response from Webshare API", level=xbmc.LOGERROR)
                return []
                
            # Parsování XML
            xml = ET.fromstring(response.content)
            if not webshare_api.is_ok(xml):
                xbmc.log("[plugin.video.milionar] Webshare API returned error", level=xbmc.LOGERROR)
                return []
                
            # Zpracování výsledků s našimi novými funkcemi
            all_items = {}
            for file in xml.iter('file'):
                ident = file.findtext('ident')
                if ident and ident not in all_items:
                    item = {e.tag: e.text for e in file}
                    item['name'] = item.get('name', '')
                    all_items[ident] = item
            
            file_items = list(all_items.values())
            
            # Seřazení podle typu obsahu - filmy vs seriály
            if self.is_episode:
                # Pro seriály/epizody použijeme search_series.py
                file_items = sort_series_files_by_priority(file_items, self.search_query)
            elif self.manual_search:
                # Pro manuální vyhledávání použijeme sort_files_by_priority s filter_series=False
                file_items = sort_files_by_priority(file_items, self.search_query, filter_series=False)
            else:
                # Pro filmy použijeme search_movies.py  
                file_items = sort_files_by_priority(file_items, self.search_query, filter_series=True)
            
            # Kontrola počtu výsledků
            xbmc.log(f"[plugin.video.milionar] Found {len(file_items)} search results", level=xbmc.LOGINFO)
            
            # Pokud nejsou žádné výsledky, zkusíme alternativní dotaz
            if not file_items and self.movie_title and self.movie_year:
                xbmc.log("[plugin.video.milionar] No results found, trying alternative query", level=xbmc.LOGINFO)
                
                alt_query = self.movie_title
                
                response = webshare_api.api_call('search', {
                    'what': alt_query,
                    'category': 'video',
                    'sort': '',
                    'limit': 50,
                    'offset': 0,
                    'wst': token
                })
                
                xml = ET.fromstring(response.content)
                if webshare_api.is_ok(xml):
                    alt_items = {}
                    for file in xml.iter('file'):
                        ident = file.findtext('ident')
                        if ident and ident not in alt_items:
                            item = {e.tag: e.text for e in file}
                            item['name'] = item.get('name', '')
                            alt_items[ident] = item
                    
                    file_items = list(alt_items.values())
                    # Seřazení alternativních výsledků podle typu obsahu
                    if self.is_episode:
                        file_items = sort_series_files_by_priority(file_items, alt_query)
                    else:
                        file_items = sort_files_by_priority(file_items, alt_query, filter_series=True)
                        
                xbmc.log(f"[plugin.video.milionar] Found {len(file_items)} results after alternative query", level=xbmc.LOGINFO)
            
            return file_items
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error searching Webshare: {e}", level=xbmc.LOGERROR)
            import traceback
            traceback.print_exc()
            return []

    def _populate_results_list(self):
        """Naplní seznam výsledků s extrahovanými metadaty."""
        try:
            results_list = self.getControl(50)
            results_list.reset()
            
            # NOVÉ: Přeřazení podle preferované kvality pokud je nastavena
            if self.preferred_quality:
                def quality_score(item):
                    filename = item.get('name', '')
                    file_quality = extract_quality(filename)
                    if file_quality and file_quality.lower() == self.preferred_quality.lower():
                        return 1000  # Vysoký bonus pro stejnou kvalitu
                    return 0
                
                # Seřadit podle kvality a pak původního pořadí
                self.search_results.sort(key=lambda item: quality_score(item), reverse=True)
            
            for item in self.search_results:
                filename = item.get('name', '')
                
                # Extrakce metadat z názvu souboru pomocí našich funkcí
                clean_title = extract_clean_title(filename)
                quality = extract_quality(filename)
                
                # Kontrola velikostních limitů - přeskočit soubory s neodpovídající velikostí
                if not is_valid_file_size(quality, item.get('size')):
                    continue  # Přeskočit tento soubor
                
                quality_colored = format_quality_with_color(quality)  # Barevná kvalita
                
                # NOVÉ: Zvýraznit preferovanou kvalitu
                if self.preferred_quality and quality and quality.lower() == self.preferred_quality.lower():
                    quality_colored = f"⭐ {quality_colored}"  # Přidat hvězdičku pro stejnou kvalitu
                
                language = extract_language(filename)
                file_format = extract_format(filename)
                size_formatted = format_size(item.get('size'))
                user_rating = calculate_rating(item)
                
                # Vytvoření ListItem - VÝBĚR TYPU NÁZVU:
                # MOŽNOST 1: Celý název souboru
                list_item = xbmcgui.ListItem(label=f"[COLOR 99FFFFFF]Webshare.cz: [/COLOR][COLOR FFFFFFFF]{filename}[/COLOR]")
                # MOŽNOST 2: Čitelný název filmu (zakomentováno)
                # list_item = xbmcgui.ListItem(label=clean_title)
                
                # Nastavení vlastností pro dvojřádkový design
                list_item.setProperty('quality', quality_colored)  # Používáme barevnou verzi
                list_item.setProperty('language', language)
                list_item.setProperty('format', file_format)
                list_item.setProperty('size_formatted', size_formatted)
                list_item.setProperty('user_rating', user_rating)
                
                # Původní Webshare metadata
                list_item.setProperty('ident', item.get('ident', ''))
                list_item.setProperty('filename', filename)  # Přidaný čistý název souboru
                list_item.setProperty('positive_votes', str(item.get('positive_votes', '0')))
                list_item.setProperty('negative_votes', str(item.get('negative_votes', '0')))
                
                # Přidání do seznamu
                results_list.addItem(list_item)
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error populating search results: {e}", level=xbmc.LOGERROR)
            import traceback
            traceback.print_exc()

    def _update_autoplay_label(self):
        """Aktualizuje label autoplay buttonu podle aktuálního stavu."""
        if self.is_episode:
            # Logika pro epizody
            if self.from_library:
                # Text pro epizody z knihovny
                if self.autoplay_enabled:
                    label = "Automatické přehrávání další epizody: [COLOR FF4CAF50]ZAPNUTO[/COLOR]"
                else:
                    label = "Automatické přehrávání další epizody: [COLOR FFF44336]VYPNUTO[/COLOR]"
            else:
                # Text pro epizody z menu doplnku
                label = "Automatické přehrávání dalších epizod seriálu je dostupné pouze z knihovny Kodi"
        else:
            # Text pro filmy
            label = "Automatické přehrávání další epizody seriálu je dostupné pouze pro seriály z knihovny Kodi"
        
        self.setProperty('WCS.Autoplay.Label', label)
        xbmc.log(f"[plugin.video.milionar] Autoplay label updated: {label}", level=xbmc.LOGDEBUG)

    def onClick(self, controlId):
        """Obsluha kliknutí na kontrolky."""
        if controlId == 50:  # Seznam výsledků 
            results_list = self.getControl(50)
            selected_item = results_list.getSelectedItem()
            if selected_item:
                ident = selected_item.getProperty('ident')
                name = selected_item.getProperty('filename')  # Použití čistého názvu souboru
                self.selected_result = {'ident': ident, 'name': name}
                title = self.getProperty('WCS.Search.Title')
                year = self.getProperty('WCS.Search.Year')
                plot = self.getProperty('WCS.Search.Plot')
                poster = self.getProperty('WCS.Search.Poster')
                rating = self.getProperty('WCS.Search.Rating')
                genre = self.getProperty('WCS.Search.Genre')
                director = self.getProperty('WCS.Search.Director')
                runtime = self.getProperty('WCS.Search.Runtime')
                
                # Zpracování autoplay pro epizody
                if self.is_episode:
                    if self.autoplay_enabled:
                        self.autoplay_manager.set_autoplay_enabled(True)
                        self.autoplay_manager.set_current_episode(
                            self.series_name, 
                            self.season_number, 
                            self.episode_number, 
                            self.tmdb_id
                        )
                        xbmc.log(f"[plugin.video.milionar] Autoplay enabled for {self.series_name} S{self.season_number}E{self.episode_number}", level=xbmc.LOGINFO)
                    else:
                        self.autoplay_manager.set_autoplay_enabled(False)
                
                # Uložíme jen základní informace - PlayMedia se provede v volající funkci
                self.selected_result = {'ident': ident, 'name': name}
                # Naplánovat úvodní epizodní info dialog pouze mimo knihovnu Kodi
                if self.is_episode and not self.from_library:
                    from wcs.utils import schedule_episode_info_dialog
                    schedule_episode_info_dialog(
                        series_name=self.series_name,
                        episode_title=self.episode_title or (f"Epizoda {self.episode_number}" if self.episode_number else ''),
                        season_number=self.season_number or 0,
                        episode_number=self.episode_number or 0,
                        poster=(self.episode_thumb or self.poster_url),
                        fanart=self.fanart_url,
                        rating=self.rating,
                        year=self.movie_year,
                        plot=self.plot,
                        tmdb_id=self.tmdb_id
                    )
                self.close()  # Zavřeme dialog
                
            else:
                xbmcgui.Dialog().notification("Chyba", "Nejprve vyberte soubor k přehrání", xbmcgui.NOTIFICATION_WARNING)
        elif controlId == 100:  # Tlačítko "Přehrát nejlepší"
            results_list = self.getControl(50)
            if results_list.size() > 0:
                # Nastavíme fokus na první položku v seznamu
                results_list.selectItem(0)
                # Získáme první položku ze seznamu
                first_item = results_list.getSelectedItem()
                if first_item:
                    ident = first_item.getProperty('ident')
                    name = first_item.getProperty('filename')  # Použití čistého názvu souboru
                    self.selected_result = {'ident': ident, 'name': name}
                    title = self.getProperty('WCS.Search.Title')
                    year = self.getProperty('WCS.Search.Year')
                    plot = self.getProperty('WCS.Search.Plot')
                    poster = self.getProperty('WCS.Search.Poster')
                    rating = self.getProperty('WCS.Search.Rating')
                    genre = self.getProperty('WCS.Search.Genre')
                    director = self.getProperty('WCS.Search.Director')
                    runtime = self.getProperty('WCS.Search.Runtime')
                    
                    # Zpracování autoplay pro epizody
                    if self.is_episode:
                        if self.autoplay_enabled:
                            self.autoplay_manager.set_autoplay_enabled(True)
                            self.autoplay_manager.set_current_episode(
                                self.series_name, 
                                self.season_number, 
                                self.episode_number, 
                                self.tmdb_id
                            )
                            xbmc.log(f"[plugin.video.milionar] Autoplay enabled for {self.series_name} S{self.season_number}E{self.episode_number}", level=xbmc.LOGINFO)
                        else:
                            self.autoplay_manager.set_autoplay_enabled(False)
                    
                    # Uložíme jen základní informace - PlayMedia se provede v volající funkci
                    self.selected_result = {'ident': ident, 'name': name}
                    # Naplánovat úvodní epizodní info dialog pouze mimo knihovnu Kodi
                    if self.is_episode and not self.from_library:
                        from wcs.utils import schedule_episode_info_dialog
                        schedule_episode_info_dialog(
                            series_name=self.series_name,
                            episode_title=self.episode_title or (f"Epizoda {self.episode_number}" if self.episode_number else ''),
                            season_number=self.season_number or 0,
                            episode_number=self.episode_number or 0,
                            poster=(self.episode_thumb or self.poster_url),
                            fanart=self.fanart_url,
                            rating=self.rating,
                            year=self.movie_year,
                            plot=self.plot,
                            tmdb_id=self.tmdb_id
                        )
                    self.close()  # Zavřeme dialog
                else:
                    xbmcgui.Dialog().notification("Chyba", "Nejprve vyberte soubor k přehrání", xbmcgui.NOTIFICATION_WARNING)
            else:
                xbmcgui.Dialog().notification("Chyba", "Nebyly nalezeny žádné soubory", xbmcgui.NOTIFICATION_WARNING)
        elif controlId == 101:  # Tlačítko "Stáhnout"
            self._open_download_picker()
        elif controlId == 102:  # Tlačítko "Hrát hru"
            self.start_game()
        elif controlId == 71:  # Autoplay button
            try:
                # Autoplay je dostupné pouze z knihovny
                if not self.from_library:
                    xbmcgui.Dialog().notification("Info", "Automatické přehrávání je dostupné pouze z knihovny Kodi", xbmcgui.NOTIFICATION_INFO)
                    return
                    
                # Toggle stavu pouze pro knihovnu
                self.autoplay_enabled = not self.autoplay_enabled
                self._update_autoplay_label()
                
                xbmc.log(f"[plugin.video.milionar] Autoplay button toggled: {self.autoplay_enabled}", level=xbmc.LOGINFO)
                
            except Exception as e:
                xbmc.log(f"[plugin.video.milionar] Error handling autoplay button toggle: {e}", level=xbmc.LOGERROR)

    def onAction(self, action):
        """Obsluha akcí (např. tlačítko Zpět)."""
        if action.getId() in (xbmcgui.ACTION_PREVIOUS_MENU, xbmcgui.ACTION_NAV_BACK):
            self.close()
    
    def _open_download_picker(self):
        """Otevře dialog pro výběr souboru ke stažení."""
        xbmc.log("[plugin.video.milionar] _open_download_picker: Opening download picker dialog", level=xbmc.LOGINFO)
        
        try:
            # Importujeme download dialog
            from wcs.ui.dialogs.DialogWcsDownload import show_download_dialog
            
            # Předáme všechna metadata a výsledky
            kwargs = {
                'search_results': self.search_results,
                'movie_title': self.movie_title,
                'movie_year': self.movie_year,
                'poster_url': self.poster_url,
                'rating': self.rating,
                'genre': self.genre,
                'director': self.director,
                'runtime': self.runtime,
                'plot': self.plot,
                'fanart_url': self.fanart_url,
                'is_episode': self.is_episode,
                'series_name': self.series_name,
                'season_number': self.season_number,
                'episode_number': self.episode_number,
                'episode_title': self.episode_title,
            }
            
            # Zobrazíme download picker dialog
            selected_result = show_download_dialog(**kwargs)
            
            if selected_result:
                xbmc.log(f"[plugin.video.milionar] _open_download_picker: File selected: {selected_result.get('name')}", level=xbmc.LOGINFO)
            else:
                xbmc.log("[plugin.video.milionar] _open_download_picker: No file selected", level=xbmc.LOGINFO)
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] _open_download_picker: Exception occurred: {e}", level=xbmc.LOGERROR)
            ui.popinfo(f"{self.addon.getLocalizedString(32133)}: {e}", "Chyba", icon=xbmcgui.NOTIFICATION_ERROR)

    def download_selected_file(self):
        """Stáhne vybraný soubor z výsledků vyhledávání."""
        xbmc.log("[plugin.video.milionar] download_selected_file: Starting download process", level=xbmc.LOGINFO)
        
        try:
            xbmc.log("[plugin.video.milionar] download_selected_file: Getting control 50 (results list)", level=xbmc.LOGINFO)
            # Získat vybranou položku ze seznamu
            results_list = self.getControl(50)
            xbmc.log(f"[plugin.video.milionar] download_selected_file: Control 50 result: {results_list}", level=xbmc.LOGINFO)
            
            xbmc.log("[plugin.video.milionar] download_selected_file: Getting selected item", level=xbmc.LOGINFO)
            selected_item = results_list.getSelectedItem()
            xbmc.log(f"[plugin.video.milionar] download_selected_file: Selected item: {selected_item}", level=xbmc.LOGINFO)
            
            if not selected_item:
                xbmc.log("[plugin.video.milionar] download_selected_file: No selected item, showing warning", level=xbmc.LOGINFO)
                ui.popinfo("Nejprve vyberte soubor ke stažení", "Chyba", icon=xbmcgui.NOTIFICATION_WARNING)
                return
            
            xbmc.log("[plugin.video.milionar] download_selected_file: Getting ident property", level=xbmc.LOGINFO)
            # Získat Webshare ident a název souboru
            ident = selected_item.getProperty('ident')
            xbmc.log(f"[plugin.video.milionar] download_selected_file: Ident: {ident}", level=xbmc.LOGINFO)
            
            xbmc.log("[plugin.video.milionar] download_selected_file: Getting filename property", level=xbmc.LOGINFO)
            filename = selected_item.getProperty('filename')
            xbmc.log(f"[plugin.video.milionar] download_selected_file: Filename: {filename}", level=xbmc.LOGINFO)
            
            if not ident:
                xbmc.log("[plugin.video.milionar] download_selected_file: No ident, showing error", level=xbmc.LOGINFO)
                ui.popinfo("Chybí identifikátor souboru", "Chyba", icon=xbmcgui.NOTIFICATION_ERROR)
                return
            
            # IHNED zavřít VŠECHNY dialogy po získání všech potřebných dat
            xbmc.log("[plugin.video.milionar] download_selected_file: Closing ALL dialogs immediately", level=xbmc.LOGINFO)
            
            # Zavřít všechny otevřené dialogy
            xbmc.executebuiltin("Dialog.Close(all)")
            
            # Také zavřít aktuální dialog pro jistotu
            self.close()
            
            # Spustit stahování až po zavření dialogu
            xbmc.log("[plugin.video.milionar] download_selected_file: About to import DownloadManager", level=xbmc.LOGINFO)
            from wcs.webshare import DownloadManager as download
            xbmc.log("[plugin.video.milionar] download_selected_file: DownloadManager imported successfully", level=xbmc.LOGINFO)
            
            xbmc.log(f"[plugin.video.milionar] download_selected_file: Calling download.download_file with ident: {ident}", level=xbmc.LOGINFO)
            download.download_file(ident, self.addon)
            xbmc.log("[plugin.video.milionar] download_selected_file: Download started successfully", level=xbmc.LOGINFO)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] download_selected_file: Exception occurred: {e}", level=xbmc.LOGERROR)
            xbmc.log(f"[plugin.video.milionar] download_selected_file: Exception type: {type(e)}", level=xbmc.LOGERROR)
            import traceback
            xbmc.log(f"[plugin.video.milionar] download_selected_file: Full traceback: {traceback.format_exc()}", level=xbmc.LOGERROR)
            
            xbmc.log("[plugin.video.milionar] download_selected_file: About to call ui.popinfo for error", level=xbmc.LOGINFO)
            ui.popinfo(f"Chyba při stahování: {e}", "Chyba", icon=xbmcgui.NOTIFICATION_ERROR)
            xbmc.log("[plugin.video.milionar] download_selected_file: Error notification shown", level=xbmc.LOGINFO)
    
    def start_game(self):
        """Spustí herní dialog pro film nebo epizodu seriálu."""
        try:
            # Získat metadata z properties nebo z konstruktoru
            title = self.getProperty('WCS.Search.Title')
            year = self.getProperty('WCS.Search.Year')
            plot = self.getProperty('WCS.Search.Plot')
            poster = self.getProperty('WCS.Search.Poster')
            rating = self.getProperty('WCS.Search.Rating')
            genre = self.getProperty('WCS.Search.Genre')
            director = self.getProperty('WCS.Search.Director')
            runtime = self.getProperty('WCS.Search.Runtime')
            
            # Pokud nemáme metadata z properties, použít z konstruktoru
            if not plot and hasattr(self, 'plot'):
                plot = self.plot
            if not poster and hasattr(self, 'poster_url'):
                poster = self.poster_url
            if not rating and hasattr(self, 'rating'):
                rating = self.rating
            if not genre and hasattr(self, 'genre'):
                genre = self.genre
            if not director and hasattr(self, 'director'):
                director = self.director
            if not runtime and hasattr(self, 'runtime'):
                runtime = self.runtime
            
            # Určit typ média podle dostupných dat
            media_type = "movie"  # default
            episode_title = ""
            season_number = None
            episode_number = None
            
            # Použít is_episode flag z konstruktoru pro rozlišení film vs. seriál
            if self.is_episode:
                media_type = "series"
                # Pokud máme episode_title z konstruktoru, použít ho
                if hasattr(self, 'episode_title') and self.episode_title:
                    episode_title = self.episode_title
                # Pokud máme season_number a episode_number z konstruktoru, použít je
                if hasattr(self, 'season_number') and self.season_number:
                    season_number = self.season_number
                if hasattr(self, 'episode_number') and self.episode_number:
                    episode_number = self.episode_number
                # Pokud máme series_name z konstruktoru, použít ho jako title
                if hasattr(self, 'series_name') and self.series_name:
                    title = self.series_name
            else:
                # Pro filmy zajistit, že máme správný title
                if hasattr(self, 'movie_title') and self.movie_title:
                    title = self.movie_title
                # Pro filmy také použít plot z konstruktoru pokud není v properties
                if not plot and hasattr(self, 'plot'):
                    plot = self.plot
            
            # Vytvořit fanart URL z posteru
            fanart = poster if poster else ""
            banner_fanart = fanart  # Stejně jako v router.py
            
            # Spustit herní dialog podle vzoru z router.py
            from wcs.games.DialogWcsTextGame import TextGameWindow
            
            dialog = TextGameWindow(
                "games/DialogWcsTextGame.xml",
                xbmcaddon.Addon().getAddonInfo('path'),
                "Default",
                title=title,
                fanart=fanart,
                plot=plot,
                genre=genre,
                media_type=media_type,
                banner_fanart=banner_fanart,
                episode_title=episode_title,
                season_number=season_number,
                episode_number=episode_number
            )
            
            dialog.doModal()
            del dialog
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Error starting game: {e}", level=xbmc.LOGERROR)
            xbmcgui.Dialog().notification("Chyba", f"Chyba při spouštění hry: {e}", xbmcgui.NOTIFICATION_ERROR)

# Funkce pro zobrazení dialogu vyhledávání
def show_search_dialog(search_query, movie_title='', movie_year='', poster_url='', rating='', genre='', director='', runtime='', plot='', search_results=[], is_episode=False, series_name='', season_number=None, episode_number=None, tmdb_id=None, preferred_quality=None, autoplay_request=False, fanart_url='', episode_title='', episode_thumb='', schedule_episode_info=False, from_library=False, manual_search=False):
    """
    Zobrazí dialogové okno s výsledky vyhledávání.
    
    Args:
        search_query (str): Dotaz pro vyhledávání na Webshare
        movie_title (str, optional): Název filmu pro zobrazení v hlavičce
        movie_year (str, optional): Rok filmu
        poster_url (str, optional): URL obrázku filmu pro zobrazení v dialogu
        rating (str, optional): Hodnocení filmu
        genre (str, optional): Žánr filmu
        director (str, optional): Režisér
        runtime (str, optional): Délka filmu
        plot (str, optional): Popis filmu
        search_results (list, optional): Seznam výsledků vyhledávání
        is_episode (bool, optional): Zda se jedná o epizodu seriálu
        series_name (str, optional): Název seriálu
        season_number (int, optional): Číslo sezóny
        episode_number (int, optional): Číslo epizody
        tmdb_id (int, optional): TMDb ID seriálu
        preferred_quality (str, optional): Preferovaná kvalita pro autoplay
        
    Returns:
        dict or None: Vybraný výsledek nebo None pokud uživatel nic nevybral
    """
    addon = xbmcaddon.Addon()
    
    # Dialogové okno zobrazíme pouze pokud je search_query neprázdný
    if not search_query:
        xbmcgui.Dialog().notification("Chyba", "Prázdný vyhledávací dotaz", xbmcgui.NOTIFICATION_ERROR)
        return None
        
    # OPRAVA: Pro autoplay requests nebo autoplay_first_item automaticky vybrat první soubor bez dialogu
    autoplay_first_item = addon.getSettingBool('autoplay_first_item')
    
    if autoplay_request or autoplay_first_item:
        # Pro autoplay requests zkontrolovat globální nastavení
        if autoplay_request:
            autoplay_enabled_global = addon.getSettingBool('autoplay_enabled_global')
            if not autoplay_enabled_global:
                # Autoplay je zakázán globálně - nepustit další epizodu
                xbmc.log(f"[plugin.video.milionar] Autoplay request: Autoplay disabled globally, not playing next episode", level=xbmc.LOGINFO)
                return None
        
        # Použijeme stejnou logiku jako dialog - načteme výsledky a vybereme první
        # Vytvoříme dočasný dialog pro načtení výsledků (bez zobrazení)
        temp_dialog = SearchDialog(
            "search/DialogWcsSearch.xml", 
            addon.getAddonInfo('path'),
            search_query=search_query,
            movie_title=movie_title,
            movie_year=movie_year,
            poster_url=poster_url,
            rating=rating,
            genre=genre,
            director=director,
            runtime=runtime,
            plot=plot,
            search_results=search_results,
            is_episode=is_episode,
            series_name=series_name,
            season_number=season_number,
            episode_number=episode_number,
            tmdb_id=tmdb_id,
            preferred_quality=preferred_quality,
            fanart_url=fanart_url,
            episode_title=episode_title,
            episode_thumb=episode_thumb,
            from_library=from_library,
            manual_search=manual_search,
        )
        
        # Načteme výsledky bez zobrazení dialogu
        temp_dialog.search_results = temp_dialog._perform_search()
        if temp_dialog.search_results:
            # NOVÉ: Pro autoplay_first_item zkontrolovat preferovanou kvalitu
            selected_file = None
            
            if autoplay_first_item and not autoplay_request:
                # Získat preferovanou kvalitu z nastavení
                quality_setting = addon.getSettingInt('autoplay_preferred_quality')
                quality_map = {
                    0: None,        # Nejlepší dostupná (automaticky)
                    1: ['REMUX', '4K'],  # 4K / REMUX
                    2: ['1080p'],   # 1080p
                    3: ['720p'],    # 720p
                    4: ['480p'],    # 480p
                    5: ['360p'],    # 360p
                    6: ['SD']       # SD
                }
                
                preferred_qualities = quality_map.get(quality_setting)
                
                if preferred_qualities:
                    # Hledat první soubor s preferovanou kvalitou
                    from wcs.utils import extract_quality
                    for file_item in temp_dialog.search_results:
                        file_quality = extract_quality(file_item.get('name', ''))
                        if file_quality in preferred_qualities:
                            selected_file = file_item
                            xbmc.log(f"[plugin.video.milionar] Autoplay first item: Found preferred quality {file_quality}: {file_item.get('name')}", level=xbmc.LOGINFO)
                            break
                
                # Pokud nebyl nalezen soubor s preferovanou kvalitou, vzít první dostupný
                if not selected_file:
                    selected_file = temp_dialog.search_results[0]
                    xbmc.log(f"[plugin.video.milionar] Autoplay first item: Preferred quality not found, using first available: {selected_file.get('name')}", level=xbmc.LOGINFO)
            else:
                # Pro autoplay_request použít první soubor (nejlepší podle řazení)
                selected_file = temp_dialog.search_results[0]
            
            selected_result = {'ident': selected_file.get('ident'), 'name': selected_file.get('name')}
            xbmc.log(f"[plugin.video.milionar] {'Autoplay' if autoplay_request else 'Autoplay first item'} request: Selected file: {selected_file.get('name')}", level=xbmc.LOGINFO)
            del temp_dialog
            return selected_result
        else:
            xbmc.log(f"[plugin.video.milionar] {'Autoplay' if autoplay_request else 'Autoplay first item'} request: No files found", level=xbmc.LOGINFO)
            del temp_dialog
            return None
    
    # Vytvoření instance dialogu s potřebnými parametry
    dialog = SearchDialog(
        "search/DialogWcsSearch.xml", 
        addon.getAddonInfo('path'),
        search_query=search_query,
        movie_title=movie_title,
        movie_year=movie_year,
        poster_url=poster_url,
        rating=rating,
        genre=genre,
        director=director,
        runtime=runtime,
        plot=plot,
        search_results=search_results,
        is_episode=is_episode,
        series_name=series_name,
        season_number=season_number,
        episode_number=episode_number,
        tmdb_id=tmdb_id,
        preferred_quality=preferred_quality,
        fanart_url=fanart_url,
        episode_title=episode_title,
        episode_thumb=episode_thumb,
        from_library=from_library,
        manual_search=manual_search,
    )
    
    # Zobrazení dialogu
    dialog.doModal()
    
    # Získání výsledku a zpracování přehrávání
    result = dialog.selected_result

    # Uvolnění paměti
    del dialog

    return result 