import os
import xbmcaddon
import xbmc
import xbmcvfs
import requests
from wcs.ai.DialogAIDebugRequest import show_api_request_debug

ADDON = xbmcaddon.Addon()

OPENAI_API_KEY = ADDON.getSetting('openai_api_key')
TTS_MODEL = 'gpt-4o-mini-tts'
TTS_OUTPUT_PATH = xbmcvfs.translatePath('special://temp/tts_milionar.wav')
TTS_API_URL = 'https://api.openai.com/v1/audio/speech'

VOICE_DEFAULT = 'alloy'
INSTRUCTIONS_DEFAULT = (
    'Hlas: Dramatický, energický, s napětím a důrazem na klíčová slova. '
    'Tón: Soutěžní, motivující, s občasnými dramatickými pauzami. '
    'Výslovnost: Jasná, zřetelná, s mírným důrazem na otázky. '
    'Tempo: Střídavě svižné a zpomalené v napínavých momentech. '
    'Emoce: Nadšení, napětí, soutěžní atmosféra. '
    'Celkově působ jako zkušený televizní moderátor, který dokáže udržet diváka v napětí a dodat otázce důležitost.'
)

class OpenAITTSException(Exception):
    pass

def generate_tts(question_text, voice=VOICE_DEFAULT, style=None, instructions=None, language='cz'):
    """
    Vygeneruje TTS audio soubor s otázkou pomocí OpenAI API.
    :param question_text: Text otázky (str)
    :param voice: Název hlasu (str)
    :param style: Index stylu hlasu (int/str)
    :param instructions: (volitelně) přímo instrukce pro hlas (str)
    :param language: Jazyk (str, default 'cz')
    :return: Cesta k audio souboru (str)
    :raises: OpenAITTSException při chybě
    """
    if not OPENAI_API_KEY:
        raise OpenAITTSException('Chybí OpenAI API klíč v nastavení doplňku.')

    style_prompts_list = [
        'Hlas: Dramatický, energický, s napětím a důrazem na klíčová slova. Tón: Soutěžní, motivující, s občasnými dramatickými pauzami. Výslovnost: Jasná, zřetelná, s mírným důrazem na otázky. Tempo: Střídavě svižné a zpomalené v napínavých momentech. Emoce: Nadšení, napětí, soutěžní atmosféra. Celkově působ jako zkušený televizní moderátor, který dokáže udržet diváka v napětí a dodat otázce důležitost.',
        'Hlas: Vřelý, klidný, přátelský. Tón: Uklidňující, pozitivní, zaměřený na řešení. Výslovnost: Precizní, pomalá, s jemnými pauzami pro lepší srozumitelnost. Emoce: Empatie, jistota, podpora. Styl: Jednoduchý, srozumitelný jazyk, bez zbytečné formálnosti.',
        'Hlas: Hravý, lehce ironický, s veselým podtónem. Tón: Odlehčený, zábavný, s občasnými vtipnými intonacemi. Výslovnost: Výrazná, s důrazem na pointu. Tempo: Svižné, s krátkými pauzami pro efekt. Emoce: Radost, pobavení, nadhled. Styl: Používej jednoduché, vtipné obraty, neboj se drobné nadsázky.',
        'Hlas: Standardní, bez výrazné stylizace. Tón: Neutrální, informativní. Výslovnost: Jasná, srozumitelná. Tempo: Plynulé, střední rychlost. Emoce: Minimální, zaměřeno na srozumitelnost. Styl: Formální, bez zbytečných ozdob.',
        'Hlas: Hluboký, tajemný, s dramatickými pauzami. Tón: Napínavý, lehce zlověstný, budující atmosféru. Výslovnost: Důrazná, s prodloužením klíčových slov. Tempo: Pomalé, s dlouhými pauzami pro napětí. Emoce: Tajemství, napětí, očekávání. Styl: Používej krátké věty, buduj napětí, jako bys vyprávěl detektivní příběh.'
    ]
    prompt = INSTRUCTIONS_DEFAULT
    if instructions:
        prompt = instructions
    elif style is not None:
        try:
            idx = int(style)
            if 0 <= idx < len(style_prompts_list):
                prompt = style_prompts_list[idx]
        except Exception:
            pass
    # Jinak výchozí

    headers = {
        'Authorization': f'Bearer {OPENAI_API_KEY}',
        'Content-Type': 'application/json'
    }
    data = {
        'model': TTS_MODEL,
        'input': question_text,
        'voice': voice,
        'instructions': prompt,
        'response_format': 'wav'
    }
    # Debug dialog před odesláním
    if not show_api_request_debug("OpenAI TTS", TTS_MODEL, data):
        raise OpenAITTSException('Odeslání TTS požadavku bylo zrušeno uživatelem.')
    try:
        response = requests.post(TTS_API_URL, headers=headers, json=data, timeout=30)
        if response.status_code != 200:
            raise OpenAITTSException(f'Chyba OpenAI TTS API: {response.status_code} {response.text}')
        # Ukládej přes xbmcvfs, ne open()
        f = xbmcvfs.File(TTS_OUTPUT_PATH, 'w+b')
        f.write(response.content)
        f.close()
        return TTS_OUTPUT_PATH
    except Exception as e:
        raise OpenAITTSException(f'Chyba při generování TTS: {e}') 