"""
AI Login Dialog Handler pro plugin.video.milionar
Zobrazuje dialog pro zadání AI API klíče
"""

import xbmc
import xbmcgui
import xbmcaddon
import xbmcplugin
from wcs.ui.dialogs.guide.DialogAIGuide import AIGuideDialog
from wcs.ui.dialogs.guide.DialogRemoteGuide import RemoteGuideDialog
import os


def show_ai_login_dialog(provider=None):
    """
    Zobrazí AI login dialog.
    Vrací True pokud byl zadán klíč, False jinak.
    
    Args:
        provider: Volitelný parametr pro specifikaci poskytovatele
    """
    try:
        addon = xbmcaddon.Addon()
        addon_path = addon.getAddonInfo('path')
        xml_filename = "login/DialogAILogin.xml"
        
        dialog = AILoginDialog(xml_filename, addon_path, provider=provider)
        dialog.doModal()
        result = dialog.get_result()
        del dialog
        return result.get('api_key_entered', False)
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] Error showing AI login dialog: {e}", level=xbmc.LOGERROR)
        return False


class AILoginDialog(xbmcgui.WindowXMLDialog):
    """Dialog pro zadání AI API klíče."""
    
    def __init__(self, xml_filename, script_path, provider=None):
        super().__init__(xml_filename, script_path, "Default", "1080i")
        self.addon = xbmcaddon.Addon()
        self.provider = provider
        self.api_key_entered = False
        self.dialog_closed = False
        
    def onInit(self):
        """Inicializace dialogu."""
        try:
            # Dynamicky nastav popisný text podle poskytovatele
            current_provider = self.provider or self.addon.getSetting("ai_provider") or "mistral"
            if "mistral" in current_provider.lower():
                provider_name = "Mistral AI"
            elif "openai" in current_provider.lower():
                provider_name = "OpenAI"
            elif "anthropic" in current_provider.lower():
                provider_name = "Anthropic"
            elif "google" in current_provider.lower():
                provider_name = "Google AI"
            elif "groq" in current_provider.lower():
                provider_name = "Groq"
            else:
                provider_name = "AI poskytovatel"
            # ID popisného labelu je druhý label v pořadí (pravděpodobně ID 2, ale v XML není explicitně uvedeno)
            # V Kodi XML je pořadí controlů určeno pořadím v souboru, takže použijeme getControl(2)
            try:
                label_control = self.getControl(2001)
                label_control.setLabel(f"Pro použití AI funkcí je vyžadován API klíč poskytovatele {provider_name}.")
            except Exception as e:
                xbmc.log(f"[plugin.video.milionar] AILoginDialog: Nelze nastavit popisný text: {e}", xbmc.LOGWARNING)
            pass
        except TypeError as e:
            if "function takes at most" in str(e):
                xbmc.log(f"[plugin.video.milionar] Ignored Kodi callback argument error (onInit): {e}", xbmc.LOGDEBUG)
                return
            raise
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Ignored onInit error: {e}", xbmc.LOGDEBUG)
            pass
        
    def onAction(self, action):
        """Zpracování akcí."""
        try:
            if action == xbmcgui.ACTION_NAV_BACK or action == xbmcgui.ACTION_PREVIOUS_MENU:
                self.dialog_closed = True
                self.close()
        except TypeError as e:
            if "function takes at most" in str(e):
                xbmc.log(f"[plugin.video.milionar] Ignored Kodi callback argument error (onAction): {e}", xbmc.LOGDEBUG)
                return
            raise
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Ignored onAction error: {e}", xbmc.LOGDEBUG)
            pass
            
    def onClick(self, controlId):
        """Zpracování kliknutí na tlačítka."""
        try:
            if controlId == 7101:  # Zadat klíč
                self._handle_enter_api_key()
            elif controlId == 7102:  # Návod
                self._handle_show_guide()
            elif controlId == 7103:  # Dálkové ovládání
                self._handle_show_remote_guide()
            elif controlId == 7104:  # Zavřít
                self.dialog_closed = True
                self.close()
        except TypeError as e:
            if "function takes at most" in str(e):
                xbmc.log(f"[plugin.video.milionar] Ignored Kodi callback argument error (onClick): {e}", xbmc.LOGDEBUG)
                return
            raise
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Ignored onClick error: {e}", xbmc.LOGDEBUG)
            pass
            
    def _handle_enter_api_key(self):
        """Zpracuje zadávání AI API klíče."""
        try:
            # Zjistíme aktuální poskytovatel
            current_provider = self.provider or self.addon.getSetting("ai_provider") or "mistral"
            
            # Normalizace názvu poskytovatele
            if "mistral" in current_provider.lower():
                provider_name = "Mistral AI"
                setting_key = "mistral_api_key"
            elif "openai" in current_provider.lower():
                provider_name = "OpenAI"
                setting_key = "openai_api_key"
            elif "anthropic" in current_provider.lower():
                provider_name = "Anthropic"
                setting_key = "anthropic_api_key"
            elif "google" in current_provider.lower():
                provider_name = "Google AI"
                setting_key = "google_api_key"
            elif "groq" in current_provider.lower():
                provider_name = "Groq"
                setting_key = "groq_api_key"
            else:
                provider_name = "AI"
                setting_key = "mistral_api_key"
            
            # Zadání API klíče
            keyboard = xbmc.Keyboard('', f'Zadejte {provider_name} API klíč')
            keyboard.doModal()
            
            if not keyboard.isConfirmed():
                return  # Uživatel zrušil
                
            api_key = keyboard.getText().strip()
            if not api_key:
                xbmcgui.Dialog().ok("Chyba zadání", "API klíč nemůže být prázdný.")
                return
            
            # Základní validace formátu API klíče
            if len(api_key) < 10:
                confirm = xbmcgui.Dialog().yesno(
                    "Chyba zadání",
                    f"Zadaný klíč je neobvykle krátký. API klíč pro {provider_name} obvykle obsahuje více než 10 znaků.",
                    yeslabel="Zadat znovu",
                    nolabel="Zavřít"
                )
                if confirm:
                    # Uživatel chce zadat znovu
                    self._handle_enter_api_key()
                    return
                else:
                    # Zavřít dialog bez uložení - ale NEZAVÍRAT hlavní dialog
                    return  # Pouze se vrátí zpět, dialog zůstane otevřený
            
            # Ověření klíče PŘED uložením
            if self._validate_api_key(api_key, current_provider):
                # Klíč je platný - uložit do nastavení
                self.addon.setSetting(setting_key, api_key)
                self.api_key_entered = True
                xbmcgui.Dialog().notification(
                    self.addon.getAddonInfo('name'),
                    f"API klíč pro {provider_name} byl úspěšně uložen.",
                    xbmcgui.NOTIFICATION_INFO,
                    3000
                )
                self.close()
            else:
                # Klíč se nepodařilo ověřit
                confirm = xbmcgui.Dialog().yesno(
                    "Neplatný API klíč",
                    f"Zadaný API klíč pro {provider_name} je neplatný. Zkontrolujte prosím, že jste jej zadali správně podle instrukcí poskytovatele.",
                    yeslabel="Zadat znovu",
                    nolabel="Zavřít"
                )
                if confirm:
                    # Uživatel chce zadat znovu
                    self._handle_enter_api_key()
                    return
                else:
                    # Zavřít dialog bez uložení - ale NEZAVÍRAT hlavní dialog
                    return  # Pouze se vrátí zpět, dialog zůstane otevřený
            
        except Exception as e:
            if "function takes at most" in str(e):
                xbmc.log(f"[plugin.video.milionar] Ignored callback argument error: {e}", xbmc.LOGDEBUG)
                return
            xbmc.log(f"[plugin.video.milionar] AILoginDialog: Error entering API key: {e}", level=xbmc.LOGERROR)
            xbmcgui.Dialog().ok("Chyba", f"Interní chyba při ověřování API klíče pro {provider_name}. Kontaktujte vývojáře.")
            
    def _validate_api_key(self, api_key, provider):
        """Ověří platnost AI API klíče testem volání."""
        try:
            # Dočasně uložit klíč pro ověření
            temp_setting_key = None
            if "mistral" in provider.lower():
                temp_setting_key = "mistral_api_key"
                from wcs.ai.MistralClient import validate_api_key
            elif "openai" in provider.lower():
                temp_setting_key = "openai_api_key"
                from wcs.ai.OpenAIClient import validate_api_key
            elif "anthropic" in provider.lower():
                temp_setting_key = "anthropic_api_key"
                from wcs.ai.AnthropicClient import validate_api_key
            elif "google" in provider.lower():
                temp_setting_key = "google_api_key"
                from wcs.ai.GoogleClient import validate_api_key
            elif "groq" in provider.lower():
                temp_setting_key = "groq_api_key"
                from wcs.ai.GroqClient import validate_api_key
            else:
                return False
            
            # Dočasně uložit klíč pro ověření
            original_key = self.addon.getSetting(temp_setting_key)
            self.addon.setSetting(temp_setting_key, api_key)
            
            try:
                # Ověřit klíč
                result = validate_api_key()
                return result
            finally:
                # Obnovit původní klíč (pokud existoval)
                if original_key:
                    self.addon.setSetting(temp_setting_key, original_key)
                else:
                    # Pokud původní klíč neexistoval, vyčistit nastavení
                    self.addon.setSetting(temp_setting_key, "")
                    
        except TypeError as e:
            xbmc.log(f"[plugin.video.milionar] AILoginDialog: Chybný podpis validate_api_key: {e}", level=xbmc.LOGERROR)
            xbmcgui.Dialog().ok("Chyba", f"Interní chyba při ověřování API klíče pro {provider}.")
            return False
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] AILoginDialog: Error validating API key: {e}", level=xbmc.LOGWARNING)
            return False
            
    def _handle_show_guide(self):
        """Zobrazí návod pro získání AI API klíče."""
        try:
            # Otevření jednotného guide dialogu
            guide_dialog = AIGuideDialog(
                'guide/DialogAIGuide.xml',
                self.addon.getAddonInfo('path')
            )
            guide_dialog.doModal()
            del guide_dialog
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] AILoginDialog: Error showing guide: {e}", level=xbmc.LOGERROR)
            
            # Fallback - zobrazit text přímo
            current_provider = self.addon.getSetting("ai_provider") or "mistral"
            
            # Normalizace názvu poskytovatele
            if "mistral" in current_provider.lower():
                provider_name = "Mistral AI"
                guide_text = (
                    "NÁVOD - MISTRAL AI API KLÍČ:\\n\\n"
                    "1. Přejděte na https://console.mistral.ai/\\n"
                    "2. Zaregistrujte se nebo se přihlašte\\n"
                    "3. Přejděte do sekce 'API Keys'\\n"
                    "4. Klikněte na 'Create new key'\\n"
                    "5. Zadejte název klíče a klikněte 'Create'\\n"
                    "6. Zkopírujte zobrazený API klíč\\n"
                    "7. Vložte klíč do tohoto dialogu\\n\\n"
                    "Mistral AI nabízí bezplatnou úroveň pro testování."
                )
            elif "openai" in current_provider.lower():
                provider_name = "OpenAI"
                guide_text = (
                    "NÁVOD - OPENAI API KLÍČ:\\n\\n"
                    "1. Přejděte na https://platform.openai.com/\\n"
                    "2. Zaregistrujte se nebo se přihlašte\\n"
                    "3. Přejděte do sekce 'API Keys'\\n"
                    "4. Klikněte na 'Create new secret key'\\n"
                    "5. Zadejte název klíče a klikněte 'Create'\\n"
                    "6. Zkopírujte zobrazený API klíč\\n"
                    "7. Vložte klíč do tohoto dialogu\\n\\n"
                    "OpenAI nabízí bezplatný kredit pro nové uživatele."
                )
            elif "anthropic" in current_provider.lower():
                provider_name = "Anthropic"
                guide_text = (
                    "NÁVOD - ANTHROPIC API KLÍČ:\\n\\n"
                    "1. Přejděte na https://console.anthropic.com/\\n"
                    "2. Zaregistrujte se nebo se přihlašte\\n"
                    "3. Přejděte do sekce 'API Keys'\\n"
                    "4. Klikněte na 'Create Key'\\n"
                    "5. Zadejte název klíče a klikněte 'Create'\\n"
                    "6. Zkopírujte zobrazený API klíč\\n"
                    "7. Vložte klíč do tohoto dialogu\\n\\n"
                    "Anthropic nabízí bezplatný kredit pro nové uživatele."
                )
            elif "google" in current_provider.lower():
                provider_name = "Google AI"
                guide_text = (
                    "NÁVOD - GOOGLE AI API KLÍČ:\\n\\n"
                    "1. Přejděte na https://makersuite.google.com/app/apikey\\n"
                    "2. Přihlašte se Google účtem\\n"
                    "3. Klikněte na 'Create API key'\\n"
                    "4. Vyberte projekt nebo vytvořte nový\\n"
                    "5. Zkopírujte zobrazený API klíč\\n"
                    "6. Vložte klíč do tohoto dialogu\\n\\n"
                    "Google AI Studio nabízí bezplatnou úroveň."
                )
            elif "groq" in current_provider.lower():
                provider_name = "Groq"
                guide_text = (
                    "NÁVOD - GROQ API KLÍČ:\\n\\n"
                    "1. Přejděte na https://console.groq.com/keys\\n"
                    "2. Zaregistrujte se nebo se přihlašte\\n"
                    "3. Klikněte na 'Create API Key'\\n"
                    "4. Zadejte název klíče a klikněte 'Submit'\\n"
                    "5. Zkopírujte zobrazený API klíč\\n"
                    "6. Vložte klíč do tohoto dialogu\\n\\n"
                    "Groq nabízí extrémně rychlé generování odpovědí."
                )
            else:
                provider_name = "AI"
                guide_text = (
                    "NÁVOD - AI API KLÍČ:\\n\\n"
                    "Nejdřív v nastavení doplňku vyberte poskytovatele AI\\n"
                    "a poté se vraťte pro konkrétní návod."
                )
            
            xbmcgui.Dialog().textviewer(f"Návod - {provider_name} API klíč", guide_text)
            
    def _handle_show_remote_guide(self):
        """Zobrazí návod pro nastavení dálkového ovládání."""
        try:
            # Otevření jednotného Remote guide dialogu
            remote_guide_dialog = RemoteGuideDialog(
                'guide/DialogRemoteGuide.xml',
                self.addon.getAddonInfo('path')
            )
            remote_guide_dialog.doModal()
            del remote_guide_dialog
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] AILoginDialog: Error showing remote guide: {e}", level=xbmc.LOGERROR)
            xbmcgui.Dialog().ok("Chyba", f"Došlo k chybě při zobrazení návodu dálkového ovládání: {e}")
            
    def get_result(self):
        """Vrátí výsledek dialogu."""
        return {
            'api_key_entered': self.api_key_entered,
            'dialog_closed': self.dialog_closed
        } 



