import os
import requests
import xbmcaddon
import xbmc
import json
import time
from wcs.ai.DialogAIDebugRequest import show_api_request_debug, show_api_response_debug, show_api_error_debug

# Funkce pro získání aktuálního modelu z nastavení
def get_current_model():
    """Vrací aktuálně vybraný Groq model z nastavení."""
    addon = xbmcaddon.Addon()
    selected_model = addon.getSetting("groq_model")
    if not selected_model:
        selected_model = "llama-3.3-70b-versatile"  # Výchozí model
    return selected_model

# Funkce pro odeslání promptu na Groq
def send_prompt(prompt: str, model: str = None, **kwargs) -> (str, dict):
    # Načtení nastavení addonu
    addon = xbmcaddon.Addon()
    API_KEY = addon.getSetting("groq_api_key")
    
    if not API_KEY:
        raise ValueError("Groq API klíč není nastaven.")
    
    # Extra data pro debug
    context_titles = kwargs.get('context_titles')
    context_limit = kwargs.get('context_limit')
    
    # Pokud není model specifikován, použij model z nastavení
    if model is None:
        model = get_current_model()
    
    # Načtení parametrů z nastavení
    temperature = float(addon.getSetting("groq_temperature") or "0.7")
    top_p = float(addon.getSetting("groq_top_p") or "1.0")
    max_tokens = int(addon.getSetting("groq_max_tokens") or "2048")
    
    url = "https://api.groq.com/openai/v1/chat/completions"
    headers = {
        "Authorization": f"Bearer {API_KEY}",
        "Content-Type": "application/json",
    }
    
    payload = {
        "model": model,
        "messages": [
            {"role": "user", "content": prompt}
        ],
        "temperature": temperature,
        "top_p": top_p,
        "max_tokens": max_tokens
    }
    
    xbmc.log(f"[wcs][groq] URL: {url}", getattr(xbmc, 'LOGNOTICE', xbmc.LOGINFO))
    xbmc.log(f"[wcs][groq] Model: {model}", getattr(xbmc, 'LOGNOTICE', xbmc.LOGINFO))
    
    # Debug: Zobrazí raw request před odesláním
    if not show_api_request_debug("Groq", model, payload, context_titles=context_titles, context_limit=context_limit):
        raise Exception("Odeslání API requestu bylo zrušeno uživatelem.")
    
    start_time = time.time()
    response = requests.post(url, headers=headers, json=payload, timeout=120)
    end_time = time.time()
    latency = end_time - start_time
    xbmc.log(f"[wcs][groq] Raw response: {response.text}", getattr(xbmc, 'LOGNOTICE', xbmc.LOGINFO))
    
    # Debug: Zobrazí raw response po obdržení
    try:
        response_data = response.json()
        show_api_response_debug("Groq", model, response_data, latency=latency)
    except:
        show_api_response_debug("Groq", model, {"raw_text": response.text}, latency=latency)
    
    response.raise_for_status()
    data = response.json()
    
    # Extract usage data
    usage = data.get("usage", {})
    
    # Extract content
    content = data.get("choices", [{}])[0].get("message", {}).get("content", "")
    
    return content, usage

def validate_api_key():
    """
    Ověří platnost API klíče.
    
    Returns:
        bool: True pokud je klíč platný, False jinak
    """
    addon = xbmcaddon.Addon()
    API_KEY = addon.getSetting("groq_api_key")
    
    if not API_KEY:
        return False
    
    try:
        # Jednoduché volání pro ověření klíče
        url = "https://api.groq.com/openai/v1/models"
        headers = {
            "Authorization": f"Bearer {API_KEY}",
            "Content-Type": "application/json",
        }
        response = requests.get(url, headers=headers, timeout=10)
        response.raise_for_status()
        return True
    except requests.exceptions.HTTPError as e:
        if e.response.status_code == 401:
            xbmc.log("[wcs][groq_api] validate_api_key: Neplatný API klíč (401)", xbmc.LOGWARNING)
            return False
        else:
            xbmc.log(f"[wcs][groq_api] validate_api_key: HTTP chyba {e.response.status_code}", xbmc.LOGWARNING)
            return False
    except requests.exceptions.RequestException as e:
        xbmc.log(f"[wcs][groq_api] validate_api_key: Síťová chyba - {e}", xbmc.LOGWARNING)
        return False
    except Exception as e:
        xbmc.log(f"[wcs][groq_api] validate_api_key: Neočekávaná chyba - {e}", xbmc.LOGWARNING)
        return False
