"""
Google AI (Gemini) API handler
Kompletní implementace pro Google AI Gemini API podle oficiální dokumentace
"""

import os
import requests
import xbmcaddon
import xbmc
import json
from wcs.ai.DialogAIDebugRequest import show_api_request_debug, show_api_response_debug, show_api_error_debug

# Načtení nastavení addonu
addon = xbmcaddon.Addon()
API_KEY = addon.getSetting("google_api_key")
API_BASE_URL = "https://generativelanguage.googleapis.com"
API_VERSION = "v1beta"

def get_current_model():
    """Vrací aktuálně vybraný Google AI model z nastavení."""
    addon = xbmcaddon.Addon()
    selected_model = addon.getSetting("google_model")
    if not selected_model:
        selected_model = "gemini-2.5-flash"  # Výchozí model
    return selected_model

def send_prompt(prompt, model=None):
    """
    Odešle prompt přes Google AI (Gemini) API.
    
    Args:
        prompt (str): Text prompt k odeslání
        model (str, optional): Název modelu. Pokud není specifikován, použije se model z nastavení.
    
    Returns:
        tuple: (success: bool, response_text: str)
    """
    addon = xbmcaddon.Addon()
    api_key = addon.getSetting("google_api_key")
    
    if not api_key:
        xbmc.log("[wcs][google_api] Google AI API klíč není nastaven", xbmc.LOGWARNING)
        return False, "Google AI API klíč není nastaven. Nastavte jej v nastavení doplňku."
    
    if not model:
        model = get_current_model()
    
    # Sestavení URL pro Google AI API
    url = f"{API_BASE_URL}/{API_VERSION}/models/{model}:generateContent"
    
    # Hlavičky požadavku
    headers = {
        "Content-Type": "application/json"
    }
    
    # Tělo požadavku podle Google AI API dokumentace
    data = {
        "contents": [
            {
                "parts": [
                    {
                        "text": prompt
                    }
                ]
            }
        ],
        "generationConfig": {
            "temperature": float(addon.getSetting("google_temperature") or "1.0"),
            "topP": float(addon.getSetting("google_top_p") or "0.95"),
            "topK": int(addon.getSetting("google_top_k") or "40"),
            "maxOutputTokens": int(addon.getSetting("google_max_tokens") or "2048"),
            "candidateCount": 1
        }
    }
    
    # Přidání API klíče do URL parametrů
    url_with_key = f"{url}?key={api_key}"
    
    xbmc.log(f"[wcs][google_api] Odesílání požadavku na: {url}", xbmc.LOGINFO)
    xbmc.log(f"[wcs][google_api] Model: {model}", xbmc.LOGINFO)
    xbmc.log(f"[wcs][google_api] Prompt délka: {len(prompt)} znaků", xbmc.LOGINFO)
    
    # Debug dialog pro zobrazení požadavku (pokud je povolený)
    if addon.getSetting("debug_api_requests") == "true":
        if not show_api_request_debug("Google AI API", url_with_key, data):
            raise Exception("Odeslání API requestu bylo zrušeno uživatelem.")
    
    try:
        # Odeslání požadavku
        response = requests.post(url_with_key, headers=headers, json=data, timeout=120)
        
        xbmc.log(f"[wcs][google_api] Status kód odpovědi: {response.status_code}", xbmc.LOGINFO)
        
        # Debug dialog pro zobrazení odpovědi (pokud je povolený)
        if addon.getSetting("debug_api_responses") == "true":
            show_api_response_debug("Google AI API", response.status_code, response.text)
        
        if response.status_code == 200:
            result = response.json()
            
            # Extrakce textu odpovědi podle Google AI API struktury
            if 'candidates' in result and len(result['candidates']) > 0:
                candidate = result['candidates'][0]
                if 'content' in candidate and 'parts' in candidate['content']:
                    parts = candidate['content']['parts']
                    if len(parts) > 0 and 'text' in parts[0]:
                        response_text = parts[0]['text']
                        
                        # Logování úspěchu
                        xbmc.log(f"[wcs][google_api] Úspěšná odpověď délka: {len(response_text)} znaků", xbmc.LOGINFO)
                        
                        # Logování použití tokenů (pokud je k dispozici)
                        if 'usageMetadata' in result:
                            usage = result['usageMetadata']
                            prompt_tokens = usage.get('promptTokenCount', 0)
                            completion_tokens = usage.get('candidatesTokenCount', 0)
                            total_tokens = usage.get('totalTokenCount', 0)
                            xbmc.log(f"[wcs][google_api] Token usage - Prompt: {prompt_tokens}, Response: {completion_tokens}, Total: {total_tokens}", xbmc.LOGINFO)
                        
                        return True, response_text
                    else:
                        xbmc.log("[wcs][google_api] Odpověď neobsahuje text v očekávané struktuře", xbmc.LOGWARNING)
                        return False, "Odpověď z Google AI API neobsahuje text."
                else:
                    xbmc.log("[wcs][google_api] Neplatná struktura odpovědi - chybí content", xbmc.LOGWARNING)
                    return False, "Neplatná struktura odpovědi z Google AI API."
            else:
                # Kontrola na safety blocking
                if 'candidates' in result and len(result['candidates']) > 0:
                    candidate = result['candidates'][0]
                    if 'finishReason' in candidate:
                        finish_reason = candidate['finishReason']
                        if finish_reason in ['SAFETY', 'RECITATION']:
                            xbmc.log(f"[wcs][google_api] Odpověď byla zablokována z důvodu: {finish_reason}", xbmc.LOGWARNING)
                            return False, f"Odpověď byla zablokována Google AI z bezpečnostních důvodů: {finish_reason}"
                
                xbmc.log("[wcs][google_api] Odpověď neobsahuje kandidáty", xbmc.LOGWARNING)
                return False, "Google AI API nevrátilo žádné kandidáty odpovědi."
                
        elif response.status_code == 400:
            # Chyba v požadavku
            error_detail = "Neplatný požadavek"
            try:
                error_response = response.json()
                if 'error' in error_response:
                    error_detail = error_response['error'].get('message', error_detail)
            except:
                pass
            
            xbmc.log(f"[wcs][google_api] Chyba 400: {error_detail}", xbmc.LOGWARNING)
            return False, f"Chyba v požadavku: {error_detail}"
            
        elif response.status_code == 401:
            # Neautorizovaný přístup
            xbmc.log("[wcs][google_api] Chyba 401: Neplatný API klíč", xbmc.LOGWARNING)
            return False, "Neplatný Google AI API klíč. Zkontrolujte nastavení."
            
        elif response.status_code == 403:
            # Zakázaný přístup
            xbmc.log("[wcs][google_api] Chyba 403: Přístup zamítnut", xbmc.LOGWARNING)
            return False, "Přístup k Google AI API byl zamítnut. Zkontrolujte oprávnění."
            
        elif response.status_code == 429:
            # Překročen limit rychlosti
            xbmc.log("[wcs][google_api] Chyba 429: Překročen limit rychlosti", xbmc.LOGWARNING)
            return False, "Překročen limit rychlosti Google AI API. Zkuste to později."
            
        elif response.status_code == 500:
            # Interní chyba serveru
            xbmc.log("[wcs][google_api] Chyba 500: Interní chyba serveru", xbmc.LOGWARNING)
            return False, "Interní chyba Google AI serveru. Zkuste to později."
            
        else:
            # Jiná chyba
            xbmc.log(f"[wcs][google_api] Neočekávaný status kód: {response.status_code}", xbmc.LOGWARNING)
            xbmc.log(f"[wcs][google_api] Odpověď: {response.text}", xbmc.LOGWARNING)
            return False, f"Neočekávaná chyba Google AI API: HTTP {response.status_code}"
            
    except requests.exceptions.Timeout:
        xbmc.log("[wcs][google_api] Timeout při požadavku", xbmc.LOGWARNING)
        if addon.getSetting("debug_api_errors") == "true":
            show_api_error_debug("Google AI API", "Timeout", "Požadavek vypršel po 60 sekundách")
        return False, "Timeout: Google AI API neodpovědělo do 60 sekund."
        
    except requests.exceptions.ConnectionError as e:
        xbmc.log(f"[wcs][google_api] Chyba připojení: {str(e)}", xbmc.LOGWARNING)
        if addon.getSetting("debug_api_errors") == "true":
            show_api_error_debug("Google AI API", "Connection Error", str(e))
        return False, "Chyba připojení k Google AI API. Zkontrolujte internetové připojení."
        
    except requests.exceptions.RequestException as e:
        xbmc.log(f"[wcs][google_api] Chyba požadavku: {str(e)}", xbmc.LOGWARNING)
        if addon.getSetting("debug_api_errors") == "true":
            show_api_error_debug("Google AI API", "Request Error", str(e))
        return False, f"Chyba při komunikaci s Google AI API: {str(e)}"
        
    except json.JSONDecodeError as e:
        xbmc.log(f"[wcs][google_api] Chyba parsování JSON: {str(e)}", xbmc.LOGWARNING)
        if addon.getSetting("debug_api_errors") == "true":
            show_api_error_debug("Google AI API", "JSON Parse Error", str(e))
        return False, "Chyba při parsování odpovědi z Google AI API."
        
    except Exception as e:
        xbmc.log(f"[wcs][google_api] Neočekávaná chyba: {str(e)}", xbmc.LOGWARNING)
        if addon.getSetting("debug_api_errors") == "true":
            show_api_error_debug("Google AI API", "Unexpected Error", str(e))
        return False, f"Neočekávaná chyba: {str(e)}"

def validate_api_key():
    """
    Ověří platnost API klíče.
    
    Returns:
        bool: True pokud je klíč platný, False jinak
    """
    addon = xbmcaddon.Addon()
    api_key = addon.getSetting("google_api_key")
    
    if not api_key:
        return False
    
    try:
        # Jednoduché volání pro ověření klíče
        success, response = test_api_connection()
        return success
    except requests.exceptions.HTTPError as e:
        if e.response.status_code == 401:
            xbmc.log("[wcs][google_api] validate_api_key: Neplatný API klíč (401)", xbmc.LOGWARNING)
            return False
        else:
            xbmc.log(f"[wcs][google_api] validate_api_key: HTTP chyba {e.response.status_code}", xbmc.LOGWARNING)
            return False
    except requests.exceptions.RequestException as e:
        xbmc.log(f"[wcs][google_api] validate_api_key: Síťová chyba - {e}", xbmc.LOGWARNING)
        return False
    except Exception as e:
        xbmc.log(f"[wcs][google_api] validate_api_key: Neočekávaná chyba - {e}", xbmc.LOGWARNING)
        return False

def test_api_connection():
    """
    Testuje připojení k Google AI API s jednoduchým promptem.
    
    Returns:
        tuple: (success: bool, message: str)
    """
    xbmc.log("[wcs][google_api] Testování připojení k Google AI API", xbmc.LOGINFO)
    
    test_prompt = "Odpověz pouze 'Google AI API funguje' a nic víc."
    success, response = send_prompt(test_prompt)
    
    if success:
        xbmc.log("[wcs][google_api] Test připojení úspěšný", xbmc.LOGINFO)
        return True, "Google AI API je funkční a připojení proběhlo úspěšně."
    else:
        xbmc.log(f"[wcs][google_api] Test připojení neúspěšný: {response}", xbmc.LOGWARNING)
        return False, f"Test Google AI API selhal: {response}"

def get_available_models():
    """
    Vrací seznam dostupných Google AI modelů.
    
    Returns:
        list: Seznam názvů dostupných modelů
    """
    # Aktuální modely podle oficiální dokumentace
    return [
        "gemini-2.5-flash",
        "gemini-2.5-pro", 
        "gemini-2.0-flash",
        "gemini-2.0-flash-lite",
        "gemini-1.5-pro",
        "gemini-1.5-flash",
        "gemini-1.5-flash-8b"
    ]

def get_model_info(model_name):
    """
    Vrací informace o konkrétním modelu.
    
    Args:
        model_name (str): Název modelu
        
    Returns:
        dict: Informace o modelu
    """
    model_info = {
        "gemini-2.5-flash": {
            "description": "Nejnovější rychlý model s vylepšenou kvalitou",
            "context_length": "1M tokens",
            "supports_thinking": True
        },
        "gemini-2.5-pro": {
            "description": "Nejnovější vysoce výkonný model s thinking capabilities",
            "context_length": "2M tokens", 
            "supports_thinking": True
        },
        "gemini-2.0-flash": {
            "description": "Rychlý multimodální model s podporou toolů",
            "context_length": "1M tokens",
            "supports_thinking": False
        },
        "gemini-2.0-flash-lite": {
            "description": "Lehčí verze gemini-2.0-flash pro rychlé úkoly",
            "context_length": "32K tokens",
            "supports_thinking": False
        },
        "gemini-1.5-pro": {
            "description": "Vysoce výkonný model s dlouhým kontextem",
            "context_length": "2M tokens",
            "supports_thinking": False
        },
        "gemini-1.5-flash": {
            "description": "Rychlý a efektivní model pro běžné úkoly",
            "context_length": "1M tokens",
            "supports_thinking": False
        },
        "gemini-1.5-flash-8b": {
            "description": "Kompaktní model pro rychlé odpovědi",
            "context_length": "1M tokens",
            "supports_thinking": False
        }
    }
    
    return model_info.get(model_name, {
        "description": "Neznámý model",
        "context_length": "Neznámá",
        "supports_thinking": False
    }) 