# wcs/ai/DialogContinueWatching.py
"""
Dialog "Pokračovat ve sledování" - Fullscreen dialog pro naposledy přehrané položky.
Dědí z AIChatRecommendationDialog a sdílí jeho XML grafiku.
"""

import xbmc
import xbmcgui
import xbmcaddon
import threading
from urllib.parse import quote_plus

from wcs.ai.DialogAIChatRecommendation import AIChatRecommendationDialog
from wcs import user_data
from wcs import utils as wcs_utils


class ContinueWatchingDialog(AIChatRecommendationDialog):
    """
    Dialog pro naposledy přehrané položky.
    Dědí všechny funkce z AI Chat dialogu, ale přetěžuje:
    - onInit() - načítá z historie místo AI inference
    - _populate_preview_actions() - upravené akční tlačítka (Další díl, Smazat)
    """
    
    def __init__(self, xml_filename, addon_path, *args, **kwargs):
        # Vyčistit vlastní parametry před předáním parentovi
        self._show_nav_on_init = kwargs.pop('show_nav_sidebar', False)
        self._nav_position = kwargs.pop('nav_position', 0)
        kwargs.pop('context', None)
        super(ContinueWatchingDialog, self).__init__(xml_filename, addon_path, *args, **kwargs)
        self._history_items = []  # Cache pro položky historie
        self._current_item_index = 0  # Index aktuální položky pro akce
        # Identifikátor sekce pro navigační sidebar
        self._nav_section_id = 'continue_watching'


    
    def onInit(self):
        """Přetížená inicializace - načte položky z historie místo AI inference."""
        # Inicializovat proměnné z parent třídy (důležité!)
        self._last_preview_id = None
        self._preview_cache = {}
        self._action_token = 0  # Token pro invalidaci timerů
        self.history = []
        self.media_type = 'movie'  # Default
        self._prompt_builder = None
        self._is_showing_history = True  # Flag pro rozlišení mezi historií a AI doporučeními
        
        # Načíst historii
        self._history_items = user_data.load_recently_played(self.addon)
        
        if not self._history_items:
            # Prázdná historie - zobrazit hlášku (podobně jako v Naposledy přehrané)
            xbmcgui.Dialog().ok(
                "Pokračovat ve sledování",
                "Historie přehrávání je prázdná.\n\nPřehrajte nějaký film nebo seriál z menu a položka se zde zobrazí."
            )
            self.close()
            return
        
        # Nastavit titulek
        self.setProperty('WCS.AIChat.Title', 'Pokračovat ve sledování')
        self.setProperty('WCS.AIChat.Status', 'Připraven')
        
        # Zobrazit uvítací zprávu v chatu
        self.append_chat_message("AI", "Vyberte titul z historie nebo si nechte doporučit podobné.")
        
        # Nastavit AI tlačítka
        self._setup_initial_buttons()
        
        # Naplnit poster flow z historie
        self._populate_from_history()
        
        # Pokud byl dialog spuštěn z navigačního sidebaru, zobrazit ho bez animace
        if getattr(self, '_show_nav_on_init', False):
            self._show_nav_sidebar(animate=False)

        else:
            # Skrýt chat sidebar na začátku, fokus na postery
            self.clearProperty('WCS.AIChat.Visible')
            
            # Krátký delay aby animace nestihla přerušit focus
            def set_focus_delayed():
                import time
                time.sleep(0.5)
                try:
                    self.setFocusId(2000)
                except:
                    pass
            
            import threading
            threading.Thread(target=set_focus_delayed, daemon=True).start()

    
    def _setup_initial_buttons(self):
        """Nastaví počáteční AI tlačítka."""
        buttons = [
            {"label": "Doporuč mi na základě historie", "value": "recommend_from_history"},
            {"label": "Doporuč podobné filmy", "value": "recommend_similar_movies"},
            {"label": "Doporuč podobné seriály", "value": "recommend_similar_series"}
        ]
        self.update_buttons(buttons)
    
    def _populate_from_history(self):
        """Naplní poster flow položkami z historie."""
        list_control = self.getControl(2000)
        list_control.reset()
        
        for item in self._history_items:
            title = item.get('title', '')
            year = item.get('year', '')
            poster = item.get('poster', '')
            media_type = item.get('media_type', 'movie')
            tmdb_id = item.get('tmdb_id', '')
            
            # Pro seriály přidáme info o epizodě do labelu
            if media_type == 'series':
                season = item.get('season')
                episode = item.get('episode')
                episode_title = item.get('episode_title', '')
                
                display_title = title
                if season is not None and episode is not None:
                    display_title = f"{title} S{int(season):02d}E{int(episode):02d}"
                    if episode_title and not episode_title.startswith('S') and episode_title != f"Epizoda {episode}":
                        display_title += f" {episode_title}"
            else:
                display_title = f"{title} ({year})" if year else title
            
            list_item = xbmcgui.ListItem(label=display_title)
            
            if poster:
                list_item.setArt({'poster': poster})
            
            # Uložit metadata pro akce
            list_item.setProperty('tmdb_id', str(tmdb_id) if tmdb_id else '')
            list_item.setProperty('media_type', media_type)
            list_item.setProperty('title', title)
            list_item.setProperty('year', str(year) if year else '')
            
            # Pro seriály uložíme i info o epizodě
            if media_type == 'series':
                list_item.setProperty('season', str(item.get('season', '')))
                list_item.setProperty('episode', str(item.get('episode', '')))
                list_item.setProperty('episode_title', item.get('episode_title', ''))
            
            list_control.addItem(list_item)
        
        # Pokud máme položky, nastavit preview pro první
        if list_control.size() > 0:
            self._update_preview_for_current_item()
    
    def _populate_preview_actions(self, media_type, tmdb_id, title, year):
        """Přetíženo - upravené akční tlačítka pro Continue Watching."""
        try:
            list_control = self.getControl(6100)
            list_control.reset()
            
            # Získat aktuální index
            poster_list = self.getControl(2000)
            current_index = poster_list.getSelectedPosition()
            
            # Rozlišit mezi položkami z historie a AI doporučeními
            # Používáme flag _is_showing_history místo porovnání indexů
            is_history_item = getattr(self, '_is_showing_history', False) and current_index >= 0 and current_index < len(self._history_items)
            
            if is_history_item:
                # ===== SCÉNÁŘ 1: Položky z historie =====
                if media_type == 'movie':
                    # Film z historie: Přehrát, Detaily, Podobné, Smazat
                    actions = [
                        {'label': 'Přehrát znovu', 'action': 'play'},
                        {'label': 'Detaily', 'action': 'details'},
                        {'label': 'Podobné', 'action': 'similar'},
                        {'label': 'Smazat', 'action': 'remove_from_history'},
                    ]
                elif media_type in ['series', 'tv']:
                    # Seriál z historie: Další epizoda, Přehrát, Sezóny, Podobné, Smazat
                    actions = [
                        {'label': 'Pokračovat', 'action': 'next_episode', 'icon': 'icons/infodialogs/play.png'},
                        {'label': 'Přehrát znovu', 'action': 'play_episode'},
                        {'label': 'Sezóny', 'action': 'seasons'},
                        {'label': 'Podobné', 'action': 'similar'},
                        {'label': 'Smazat', 'action': 'remove_from_history'},
                    ]
                else:
                    actions = [
                        {'label': 'Smazat', 'action': 'remove_from_history'},
                    ]
            else:
                # ===== SCÉNÁŘ 2: AI doporučení (kopie z AI chat dialogu) =====
                if media_type == 'movie':
                    actions = [
                        {'label': 'Přehrát', 'action': 'play'},
                        {'label': 'Detaily', 'action': 'details'},
                        {'label': 'Moje filmy', 'action': 'add_list'},
                        {'label': 'Knihovna', 'action': 'add_library'},
                        {'label': 'Podobné', 'action': 'similar'},
                    ]
                elif media_type in ['series', 'tv']:
                    actions = [
                        {'label': 'Detaily', 'action': 'details'},
                        {'label': 'Sezóny', 'action': 'seasons'},
                        {'label': 'Moje seriály', 'action': 'add_list'},
                        {'label': 'Knihovna', 'action': 'add_library'},
                        {'label': 'Podobné', 'action': 'similar'},
                    ]
                else:  # collection
                    actions = [
                        {'label': 'Filmy', 'action': 'collection_movies'},
                        {'label': 'Moje filmy', 'action': 'add_list'},
                    ]
            
            for act in actions:
                item = xbmcgui.ListItem(label=act['label'])
                item.setProperty('action', act['action'])
                item.setProperty('tmdb_id', str(tmdb_id))
                item.setProperty('title', title)
                item.setProperty('year', year)
                item.setProperty('media_type', media_type)
                item.setProperty('history_index', str(current_index))
                # Nastavit ikonu pokud existuje
                if 'icon' in act:
                    item.setArt({'icon': act['icon']})
                list_control.addItem(item)
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] ContinueWatching preview actions error: {e}", xbmc.LOGERROR)
    
    def _handle_preview_action_click(self):
        """Přetíženo - zpracování akcí specifických pro Continue Watching."""
        try:
            # Reset automatického zobrazení tlačítek (jakákoliv akce ukončí delay)
            self.clearProperty('WCS.AIPreview.ShowActions')
            
            list_control = self.getControl(6100)
            item = list_control.getSelectedItem()
            if not item:
                return
            
            action = item.getProperty('action')
            tmdb_id = item.getProperty('tmdb_id')
            title = item.getProperty('title')
            year = item.getProperty('year')
            media_type = item.getProperty('media_type')
            history_index = item.getProperty('history_index')
            
            if action == 'next_episode':
                self._handle_next_episode(title, tmdb_id, history_index)
            elif action == 'play_episode':
                self._handle_play_episode(history_index)
            elif action == 'remove_from_history':
                self._handle_remove_from_history(history_index)
            elif action == 'play':
                # Přehrát film pomocí klasického search dialogu
                self._handle_play_movie(history_index)
            elif action == 'similar':
                # Podobné - s rozlišením pro filmy a seriály
                self._handle_similar(title, year, media_type)
            else:
                # Delegovat na parent pro standardní akce (details, add_list, atd.)
                super(ContinueWatchingDialog, self)._handle_preview_action_click()
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] ContinueWatching action error: {e}", xbmc.LOGERROR)
    
    def _handle_similar(self, title, year, media_type):
        """Spustí AI doporučení podobných filmů nebo seriálů."""
        try:
            # KRITICKÉ: Nastavit správný media_type podle aktuálního titulu
            # Tím se zajistí správný system prompt (filmy vs seriály)
            if media_type in ['series', 'tv']:
                self.media_type = 'series'
                inspiration_text = f"Seriál '{title}' ({year}) mě nadchl. Hledám podobný seriál - může mít jinou atmosféru, ale ať mě to vtáhne stejně."
            else:
                self.media_type = 'movie'
                inspiration_text = f"Film '{title}' ({year}) mě zaujal. Chci zažít něco podobného. Nemusí to být stejný žánr, ale ať to má podobnou atmosféru."
            
            # Vymazat starý kontext (jde o prostý dotaz bez kontextu)
            self.context = None
            
            # Reinicializovat prompt builder s novým media_type a bez kontextu
            from wcs.ai.DialogAIChatRecommendation import AIRecommendationPromptBuilder
            self._prompt_builder = AIRecommendationPromptBuilder(self.media_type, self.context)
            
            # Skrýt overlay a fokus na chat
            self._clear_preview_properties()
            self.setFocusId(4000)
            
            # Spustit načítací animaci
            self._start_button_loading("Hledám podobné...")
            self.handle_user_input(inspiration_text)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Similar action error: {e}", xbmc.LOGERROR)
    
    def fetch_and_display_recommendations(self, recommendations):
        """Přetíženo - nastaví flag že už nezobrazujeme historii."""
        # Po AI doporučení už nezobrazujeme historii
        self._is_showing_history = False
        # Volat parent implementaci
        super(ContinueWatchingDialog, self).fetch_and_display_recommendations(recommendations)
    
    def _handle_next_episode(self, series_name, tmdb_id, history_index):
        """Přehraje další díl seriálu."""
        try:
            idx = int(history_index) if history_index else 0
            if idx < 0 or idx >= len(self._history_items):
                return
            
            history_item = self._history_items[idx]
            current_season = history_item.get('season')
            current_episode = history_item.get('episode')
            
            if current_season is None or current_episode is None:
                xbmcgui.Dialog().notification(
                    self.addon.getAddonInfo('name'),
                    "Chybí informace o aktuální epizodě",
                    xbmcgui.NOTIFICATION_WARNING
                )
                return
            
            # Použít existující funkci
            from wcs.utils import play_next_episode_with_metadata
            
            params = {
                'tmdb_id': tmdb_id,
                'current_season': current_season,
                'current_episode': current_episode,
                'series_name': series_name
            }
            
            # Zavřít dialog a přehrát
            self.close()
            play_next_episode_with_metadata(params, self.addon)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Next episode error: {e}", xbmc.LOGERROR)
    
    def _handle_play_movie(self, history_index):
        """Přehraje film z historie pomocí klasického search dialogu."""
        try:
            idx = int(history_index) if history_index else 0
            if idx < 0 or idx >= len(self._history_items):
                return
            
            history_item = self._history_items[idx]
            
            from wcs.utils import play_movie_from_addon
            
            # Sestavit meta slovník z historie
            meta = {
                'title': history_item.get('title', ''),
                'year': history_item.get('year', ''),
                'plot': history_item.get('plot', ''),
                'poster': history_item.get('poster', ''),
                'fanart': '',
                'rating': history_item.get('rating', ''),
                'genre': history_item.get('genre', ''),
                'director': '',
                'runtime': '',
                'tmdb_id': history_item.get('tmdb_id'),
            }
            
            # Zavřít dialog před přehráváním
            self.close()
            
            # Spustit klasické přehrávání (používá standardní search dialog)
            play_movie_from_addon(meta, self.addon)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Play movie error: {e}", xbmc.LOGERROR)
    
    def _handle_play_episode(self, history_index):
        """Přehraje aktuální epizodu z historie."""
        try:
            idx = int(history_index) if history_index else 0
            if idx < 0 or idx >= len(self._history_items):
                return
            
            history_item = self._history_items[idx]
            
            from wcs.utils import search_and_play_episode
            
            self.close()
            search_and_play_episode(
                series_name=history_item.get('title', ''),
                season=history_item.get('season', 1),
                episode=history_item.get('episode', 1),
                ep_name=history_item.get('episode_title', ''),
                plot=history_item.get('plot', ''),
                poster=history_item.get('poster', ''),
                fanart='',
                rating=history_item.get('rating', ''),
                genre=history_item.get('genre', ''),
                addon=self.addon,
                year=history_item.get('year', ''),
                runtime='',
                tmdb_id=history_item.get('tmdb_id', '')
            )
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Play episode error: {e}", xbmc.LOGERROR)
    
    def _handle_remove_from_history(self, history_index):
        """Smaže položku z historie."""
        try:
            idx = int(history_index) if history_index else 0
            if idx < 0 or idx >= len(self._history_items):
                return
            
            removed_item = self._history_items[idx]
            item_title = removed_item.get('title', 'Neznámý')
            
            # Potvrzovací dialog
            dialog = xbmcgui.Dialog()
            if not dialog.yesno("Smazat z historie", f"Opravdu chcete smazat '{item_title}' z historie?"):
                return
            
            # Smazat položku
            self._history_items.pop(idx)
            user_data.store_recently_played(self.addon, self._history_items)
            
            # Notifikace
            xbmcgui.Dialog().notification(
                self.addon.getAddonInfo('name'),
                f"'{item_title}' smazáno z historie",
                xbmcgui.NOTIFICATION_INFO
            )
            
            # Obnovit poster flow
            if self._history_items:
                self._populate_from_history()
            else:
                # Historie je prázdná - zavřít dialog
                self.close()
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Remove from history error: {e}", xbmc.LOGERROR)
    
    def _fetch_preview_details(self, tmdb_id, media_type):
        """Přetíženo - pro seriály načte epizodní metadata místo dat seriálu."""
        try:
            import requests
            from wcs.metadata.TMDbClient import get_tmdb_api_key
            
            # KRITICKÁ KONTROLA: Pokud nezobrazujeme historii (AI doporučení), použít parent implementaci
            if not getattr(self, '_is_showing_history', False):
                super(ContinueWatchingDialog, self)._fetch_preview_details(tmdb_id, media_type)
                return
            
            # Získat aktuální položku z historie
            poster_list = self.getControl(2000)
            current_index = poster_list.getSelectedPosition()
            
            if current_index < 0 or current_index >= len(self._history_items):
                # Fallback na parent implementaci
                super(ContinueWatchingDialog, self)._fetch_preview_details(tmdb_id, media_type)
                return
            
            history_item = self._history_items[current_index]
            
            if media_type in ['series', 'tv']:
                # Pro seriály: načíst data seriálu + epizodní specifika
                api_key = get_tmdb_api_key()
                
                season_num = history_item.get('season')
                episode_num = history_item.get('episode')
                episode_title = history_item.get('episode_title', '')
                
                # Unikátní cache klíč pro položky z historie (aby nedocházelo ke kolizi s AI doporučeními)
                cache_key = f"{tmdb_id}_history_S{season_num}E{episode_num}"
                
                # 1. Načíst data seriálu
                series_url = f'https://api.themoviedb.org/3/tv/{tmdb_id}'
                series_params = {'api_key': api_key, 'language': 'cs-CZ', 'append_to_response': 'credits'}
                series_resp = requests.get(series_url, params=series_params, timeout=5)
                
                if series_resp.status_code != 200:
                    super(ContinueWatchingDialog, self)._fetch_preview_details(tmdb_id, media_type)
                    return
                
                series_data = series_resp.json()
                
                # 2. Načíst data epizody (pro rating a still)
                episode_rating = ''
                episode_still = ''
                
                if season_num is not None and episode_num is not None:
                    episode_url = f'https://api.themoviedb.org/3/tv/{tmdb_id}/season/{season_num}/episode/{episode_num}'
                    episode_params = {'api_key': api_key, 'language': 'cs-CZ'}
                    try:
                        ep_resp = requests.get(episode_url, params=episode_params, timeout=5)
                        if ep_resp.status_code == 200:
                            ep_data = ep_resp.json()
                            if ep_data.get('vote_average'):
                                episode_rating = f"{ep_data.get('vote_average'):.1f}/10"
                            if ep_data.get('still_path'):
                                episode_still = f"https://image.tmdb.org/t/p/w1280{ep_data['still_path']}"
                    except:
                        pass
                
                # 3. Sestavit cache data
                cache_data = {
                    'rating': f"{series_data.get('vote_average', 0):.1f}/10" if series_data.get('vote_average') else '',
                    'genre': ', '.join([g['name'] for g in series_data.get('genres', [])[:3]]),
                    # POUŽÍT PLOT Z HISTORIE (popis epizody, ne seriálu!)
                    'plot': history_item.get('plot', series_data.get('overview', '')),
                    'poster': f"https://image.tmdb.org/t/p/w500{series_data.get('poster_path')}" if series_data.get('poster_path') else '',
                    # Fanart: prioritně still epizody, fallback na backdrop seriálu
                    'fanart': episode_still or (f"https://image.tmdb.org/t/p/w1280{series_data.get('backdrop_path')}" if series_data.get('backdrop_path') else ''),
                    'tagline': series_data.get('tagline', ''),
                }
                
                # Runtime: počet sezón + status
                num_seasons = series_data.get('number_of_seasons', 0)
                runtime_parts = []
                if num_seasons:
                    runtime_parts.append(f"{num_seasons} sezón")
                
                status = series_data.get('status', '')
                if status == 'Returning Series':
                    runtime_parts.append("[COLOR green]Pokračuje[/COLOR]")
                elif status in ['Ended', 'Canceled']:
                    runtime_parts.append("[COLOR red]Ukončen[/COLOR]")
                    
                cache_data['runtime'] = "  |  ".join(runtime_parts)
                
                # Tvůrci
                creators = [c['name'] for c in series_data.get('created_by', [])][:2]
                cache_data['director'] = f"Tvůrci: {', '.join(creators)}" if creators else ''
                
                # === FORMÁTOVANÉ PROPERTIES ===
                # Tagline
                if cache_data.get('tagline'):
                    cache_data['fmt_tagline'] = f"[I][COLOR FFB0B0B0]{cache_data['tagline']}[/COLOR][/I]"
                else:
                    cache_data['fmt_tagline'] = ""

                # Metadata: Year • Rating • Runtime
                meta_parts = []
                year = series_data.get('first_air_date', '')[:4]
                if year:
                    meta_parts.append(year)
                if cache_data['rating']:
                    meta_parts.append(cache_data['rating'])
                if cache_data['runtime']:
                    meta_parts.append(cache_data['runtime'])
                
                if meta_parts:
                    cache_data['fmt_metadata'] = "[COLOR grey]" + "  •  ".join(meta_parts) + "[/COLOR]"
                else:
                    cache_data['fmt_metadata'] = ""
                
                # Extended: Genre • Director
                ext_parts = []
                if cache_data['genre']:
                    ext_parts.append(cache_data['genre'])
                if cache_data['director']:
                    ext_parts.append(cache_data['director'])
                
                if ext_parts:
                    cache_data['fmt_extended'] = "[COLOR FFAAAAAA]" + "  •  ".join(ext_parts) + "[/COLOR]"
                else:
                    cache_data['fmt_extended'] = ""
                
                # === NOVÝ ŘÁDEK: EPIZODNÍ INFO (Varianta C) ===
                # Formát: Sezóna X • Epizoda Y  │  Název epizody  •  7.8/10
                if season_num is not None and episode_num is not None:
                    ep_info_parts = [f"Sezóna {season_num}", f"Epizoda {episode_num}"]
                    ep_info_str = " • ".join(ep_info_parts)
                    
                    if episode_title and episode_title != f"S{int(season_num):02d}E{int(episode_num):02d}" and episode_title != f"Epizoda {episode_num}":
                        ep_info_str += f"  │  {episode_title}"
                    
                    if episode_rating:
                        ep_info_str += f"  •  {episode_rating}"
                    
                    cache_data['fmt_episode_info'] = f"[COLOR FF00BFFF]{ep_info_str}[/COLOR]"  # DeepSkyBlue
                else:
                    cache_data['fmt_episode_info'] = ""
                
                # Flag pro layout
                cache_data['is_compact'] = "true" if not cache_data.get('tagline') else ""
                
                # Uložit do cache s unikátním klíčem (aby nedocházelo ke kolizi s AI doporučeními)
                self._preview_cache[cache_key] = cache_data
                
                # Aplikovat data okamžitě (jsme v threadu spuštěném z _apply_new_preview)
                # Kontrola zda je stále vybraná stejná položka
                try:
                    current_idx = poster_list.getSelectedPosition()
                    if current_idx == current_index:
                        self._apply_cached_preview_data(cache_data)
                except:
                    self._apply_cached_preview_data(cache_data)
            else:
                # Pro filmy: použít parent implementaci
                super(ContinueWatchingDialog, self)._fetch_preview_details(tmdb_id, media_type)
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] ContinueWatching preview fetch error: {e}", xbmc.LOGERROR)
            # Fallback na parent
            super(ContinueWatchingDialog, self)._fetch_preview_details(tmdb_id, media_type)
    
    def _apply_cached_preview_data(self, data):
        """Přetíženo - přidává WCS.AIPreview.EpisodeInfo pro seriály."""
        # Volat parent implementaci
        super(ContinueWatchingDialog, self)._apply_cached_preview_data(data)
        
        # Přidat epizodní info (pouze pro seriály)
        episode_info = data.get('fmt_episode_info', '')
        if episode_info:
            self.setProperty('WCS.AIPreview.EpisodeInfo', episode_info)
        else:
            self.clearProperty('WCS.AIPreview.EpisodeInfo')
    
    def onClick(self, controlId):
        """Přetíženo pro handling speciálních tlačítek."""
        # AI tlačítka
        if controlId == 4000:
            list_control = self.getControl(4000)
            item = list_control.getSelectedItem()
            if not item:
                return
            
            val = item.getProperty('value')
            label = item.getLabel()
            
            if val == 'recommend_from_history':
                # Spustit AI doporučení s kontextem historie (mix filmů a seriálů)
                self._start_ai_from_history('mixed')
            elif val == 'recommend_similar_movies':
                self._start_ai_from_history('movie')
            elif val == 'recommend_similar_series':
                self._start_ai_from_history('series')
            else:
                # Delegovat na parent
                super(ContinueWatchingDialog, self).onClick(controlId)
        else:
            # Delegovat na parent pro ostatní controls
            super(ContinueWatchingDialog, self).onClick(controlId)
    
    def _start_ai_from_history(self, media_filter='mixed'):
        """
        Spustí AI doporučení na základě historie.
        
        Args:
            media_filter: 'movie', 'series', nebo 'mixed'
        """
        # Filtrovat položky podle typu
        if media_filter == 'movie':
            filtered = [i for i in self._history_items if i.get('media_type') == 'movie']
            self.media_type = 'movie'
        elif media_filter == 'series':
            filtered = [i for i in self._history_items if i.get('media_type') == 'series']
            self.media_type = 'series'
        else:
            filtered = self._history_items
            self.media_type = 'movie'  # Default, ale kontext bude mixed
        
        if not filtered:
            self.append_system_message(f"V historii nejsou žádné {'filmy' if media_filter == 'movie' else 'seriály' if media_filter == 'series' else 'položky'}.")
            return
        
        # Nastavit kontext
        self.context = {
            'type': 'recently_played',
            'media_type': self.media_type if media_filter != 'mixed' else 'mixed',
            'data': filtered
        }
        
        # Inicializovat prompt builder
        from wcs.ai.DialogAIChatRecommendation import AIRecommendationPromptBuilder
        self._prompt_builder = AIRecommendationPromptBuilder(self.media_type, self.context)
        
        # Vytvořit zprávu pro uživatele
        if media_filter == 'movie':
            user_msg = "Doporuč mi filmy na základě mé historie sledování."
        elif media_filter == 'series':
            user_msg = "Doporuč mi seriály na základě mé historie sledování."
        else:
            user_msg = "Doporuč mi filmy a seriály na základě mé historie sledování."
        
        # Přidat do historie a spustit
        self.history.append({"role": "user", "content": user_msg})
        self.append_chat_message("Ty", user_msg)
        
        # Spustit loading animaci
        self._start_button_loading("Analyzuji historii...")
        
        # Zavolat AI
        self._start_ai_conversation(is_initial_context=True)


def show_continue_watching_dialog(addon, show_nav_sidebar=False, nav_position=0):
    """Zobrazí dialog Pokračovat ve sledování."""
    # Nejdřív zkontrolovat, jestli je historie neprázdná
    items = user_data.load_recently_played(addon)
    if not items:
        xbmcgui.Dialog().notification(
            addon.getAddonInfo('name'),
            "Historie je prázdná",
            xbmcgui.NOTIFICATION_INFO
        )
        return
    
    addon_path = addon.getAddonInfo('path')
    
    # Použijeme SDÍLENÝ XML z AI Chat dialogu
    dialog = ContinueWatchingDialog(
        "ai/DialogAIChatRecommendation.xml",
        addon_path,
        "Default",
        "1080i",
        show_nav_sidebar=show_nav_sidebar,
        nav_position=nav_position
    )
    dialog.doModal()
    del dialog


