# wcs/tmdb_login_dialog.py

import xbmcgui
import xbmcaddon
import xbmc
from wcs.ui.dialogs.guide.DialogTMDbGuide import TMDbGuideDialog
from wcs.ui.dialogs.guide.DialogRemoteGuide import RemoteGuideDialog
from wcs.metadata import TMDbClient as tmdb_handler


"""Login dialog pro TMDb a otevření jednotného guide dialogu."""




class TMDbLoginDialog(xbmcgui.WindowXMLDialog):
    """Dialog pro nastavení TMDb API klíče s možností zadání klíče nebo zobrazení návodu."""
    
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.addon = xbmcaddon.Addon()
        self.api_key_entered = False
        self.dialog_closed = False
        
    def onInit(self):
        """Inicializace dialogu."""
        xbmc.log("[plugin.video.milionar] TMDbLoginDialog: Dialog initialized", level=xbmc.LOGINFO)
        
    def onClick(self, controlId):
        """Obsluha kliknutí na tlačítka."""
        if controlId == 7101:  # Zadat klíč
            xbmc.log("[plugin.video.milionar] TMDbLoginDialog: Enter API key clicked", level=xbmc.LOGINFO)
            self._handle_enter_api_key()
            
        elif controlId == 7102:  # Návod
            xbmc.log("[plugin.video.milionar] TMDbLoginDialog: Guide clicked", level=xbmc.LOGINFO)
            self._handle_show_guide()
            
        elif controlId == 7103:  # Dálkové ovládání
            xbmc.log("[plugin.video.milionar] TMDbLoginDialog: Remote control clicked", level=xbmc.LOGINFO)
            self._handle_show_remote_guide()
            
        elif controlId == 7104:  # Zavřít
            xbmc.log("[plugin.video.milionar] TMDbLoginDialog: Close clicked", level=xbmc.LOGINFO)
            self.dialog_closed = True
            self.close()
            
    def _handle_enter_api_key(self):
        """Zpracuje zadávání TMDb API klíče."""
        try:
            # Zadání API klíče
            keyboard = xbmc.Keyboard('', 'Zadejte TMDb API klíč')
            keyboard.doModal()
            
            if not keyboard.isConfirmed():
                return  # Uživatel zrušil
                
            api_key = keyboard.getText().strip()
            if not api_key:
                xbmcgui.Dialog().ok("Chyba zadání", "API klíč nemůže být prázdný.")
                return

            # Základní validace formátu API klíče (TMDb klíče jsou obvykle 32 znaků)
            if len(api_key) < 20:
                confirm = xbmcgui.Dialog().yesno(
                    "Chyba zadání",
                    "Zadaný klíč je neobvykle krátký. Klíč TMDb obvykle obsahuje 32 znaků.",
                    yeslabel="Zadat znovu",
                    nolabel="Zavřít"
                )
                if confirm:
                    self._handle_enter_api_key()
                    return
                else:
                    return

            # Ověření platnosti klíče PŘED uložením
            if self._validate_api_key(api_key):
                # Klíč je platný - uložit do nastavení
                self.addon.setSetting('tmdb_api_key', api_key)
                self.addon.setSetting('tmdb_status', 'Přihlášen k TMDb')
                xbmcgui.Dialog().notification(
                    self.addon.getAddonInfo('name'),
                    "TMDb API klíč byl úspěšně ověřen a uložen.",
                    xbmcgui.NOTIFICATION_INFO,
                    3000
                )
                self.api_key_entered = True
                self.close()
            else:
                confirm = xbmcgui.Dialog().yesno(
                    "Neplatný API klíč",
                    "Zadaný API klíč pro TMDb je neplatný. Zkontrolujte prosím, že jste jej zadali přesně tak, jak je uvedený na stránce TMDb.",
                    yeslabel="Zadat znovu",
                    nolabel="Zavřít"
                )
                if confirm:
                    self._handle_enter_api_key()
                    return
                else:
                    return
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] TMDbLoginDialog: Error entering API key: {e}", level=xbmc.LOGERROR)
            xbmcgui.Dialog().ok("Chyba", "Došlo k chybě při zadávání API klíče. Zkuste to prosím znovu.")
            
    def _validate_api_key(self, api_key):
        """Ověří platnost TMDb API klíče přes TMDbClient; zachová chování z nastavení.

        - Vrací False pouze při HTTP 401 (neplatný klíč)
        - Ostatní chyby vrací True (neblokující)
        - Klíč se zde neukládá natrvalo; pouze dočasně pro validaci
        """
        try:
            # Ověříme předaný klíč bez jakékoliv změny settings
            return tmdb_handler.check_tmdb_api_key_validity(self.addon, api_key=api_key)
        except Exception as e:
            # Pro zachování současné logiky považujeme jiné chyby za neblokující
            xbmc.log(f"[plugin.video.milionar] TMDbLoginDialog: validate_api_key fallback True due to: {e}", level=xbmc.LOGDEBUG)
            return True
        finally:
            # Nic nevracíme do settings – uložení proběhne až po úspěšné validaci mimo tuto funkci
            pass
            
    def _handle_show_guide(self):
        """Zobrazí návod pro získání TMDb API klíče."""
        try:
            # Otevření jednotného guide dialogu
            guide_dialog = TMDbGuideDialog(
                'guide/DialogTMDbGuide.xml',
                self.addon.getAddonInfo('path')
            )
            guide_dialog.doModal()
            del guide_dialog
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] TMDbLoginDialog: Error showing guide: {e}", level=xbmc.LOGERROR)
            xbmcgui.Dialog().ok("Chyba", f"Došlo k chybě při zobrazení návodu: {e}")
            
    def _handle_show_remote_guide(self):
        """Zobrazí návod pro nastavení dálkového ovládání."""
        try:
            # Otevření jednotného Remote guide dialogu
            remote_guide_dialog = RemoteGuideDialog(
                'guide/DialogRemoteGuide.xml',
                self.addon.getAddonInfo('path')
            )
            remote_guide_dialog.doModal()
            del remote_guide_dialog
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] TMDbLoginDialog: Error showing remote guide: {e}", level=xbmc.LOGERROR)
            xbmcgui.Dialog().ok("Chyba", f"Došlo k chybě při zobrazení návodu dálkového ovládání: {e}")
            
    def get_result(self):
        """Vrátí výsledek dialogu."""
        return {
            'api_key_entered': self.api_key_entered,
            'dialog_closed': self.dialog_closed
        }


def show_tmdb_login_dialog():
    """Zobrazí dialog pro nastavení TMDb API klíče."""
    try:
        addon = xbmcaddon.Addon()
        dialog = TMDbLoginDialog(
            'login/DialogTMDbLogin.xml',
            addon.getAddonInfo('path'),
            'Default',
            '1080i'
        )
        dialog.doModal()
        result = dialog.get_result()
        del dialog
        return result
        
    except Exception as e:
        xbmc.log(f"[plugin.video.milionar] show_tmdb_login_dialog: Error: {e}", level=xbmc.LOGERROR)
        return {'api_key_entered': False, 'dialog_closed': True} 