# wcs/ai/DialogContinueWatching.py
"""
Dialog "Pokračovat ve sledování" - Fullscreen dialog pro naposledy přehrané položky.
Dědí z AIChatRecommendationDialog a sdílí jeho XML grafiku.
"""

import xbmc
import xbmcgui
import xbmcaddon
import threading
from urllib.parse import quote_plus

from wcs.ai.DialogAIChatRecommendation import AIChatRecommendationDialog
from wcs import user_data
from wcs import utils as wcs_utils


class ContinueWatchingDialog(AIChatRecommendationDialog):
    """
    Dialog pro naposledy přehrané položky.
    Dědí všechny funkce z AI Chat dialogu, ale přetěžuje:
    - onInit() - načítá z historie místo AI inference
    - _populate_preview_actions() - upravené akční tlačítka (Další díl, Smazat)
    """
    
    def __init__(self, xml_filename, addon_path, *args, **kwargs):
        # Předáme kwargs do parent, ale vyčistíme context protože ho nastavíme sami
        kwargs.pop('context', None)
        super(ContinueWatchingDialog, self).__init__(xml_filename, addon_path, *args, **kwargs)
        self._history_items = []  # Cache pro položky historie
        self._current_item_index = 0  # Index aktuální položky pro akce
    
    def onInit(self):
        """Přetížená inicializace - načte položky z historie místo AI inference."""
        # Inicializovat proměnné z parent třídy (důležité!)
        self._last_preview_id = None
        self._preview_cache = {}
        self.history = []
        self.media_type = 'movie'  # Default
        self._prompt_builder = None
        
        # Načíst historii
        self._history_items = user_data.load_recently_played(self.addon)
        
        if not self._history_items:
            # Prázdná historie - zavřít dialog (položka by se neměla ani zobrazit v menu)
            xbmcgui.Dialog().notification(
                self.addon.getAddonInfo('name'),
                "Historie je prázdná",
                xbmcgui.NOTIFICATION_INFO
            )
            self.close()
            return
        
        # Nastavit titulek
        self.setProperty('WCS.AIChat.Title', 'Pokračovat ve sledování')
        self.setProperty('WCS.AIChat.Status', 'Připraven')
        
        # Zobrazit uvítací zprávu v chatu
        self.append_chat_message("AI", "Vyberte titul z historie nebo si nechte doporučit podobné.")
        
        # Nastavit AI tlačítka
        self._setup_initial_buttons()
        
        # Naplnit poster flow z historie
        self._populate_from_history()
        
        # Skrýt chat sidebar na začátku, fokus na postery
        self.clearProperty('WCS.AIChat.Visible')
        self.setFocusId(2000)
    
    def _setup_initial_buttons(self):
        """Nastaví počáteční AI tlačítka."""
        buttons = [
            {"label": "Doporuč mi na základě historie", "value": "recommend_from_history"},
            {"label": "Doporuč podobné filmy", "value": "recommend_similar_movies"},
            {"label": "Doporuč podobné seriály", "value": "recommend_similar_series"}
        ]
        self.update_buttons(buttons)
    
    def _populate_from_history(self):
        """Naplní poster flow položkami z historie."""
        list_control = self.getControl(2000)
        list_control.reset()
        
        for item in self._history_items:
            title = item.get('title', '')
            year = item.get('year', '')
            poster = item.get('poster', '')
            media_type = item.get('media_type', 'movie')
            tmdb_id = item.get('tmdb_id', '')
            
            # Pro seriály přidáme info o epizodě do labelu
            if media_type == 'series':
                season = item.get('season')
                episode = item.get('episode')
                episode_title = item.get('episode_title', '')
                
                display_title = title
                if season is not None and episode is not None:
                    display_title = f"{title} S{int(season):02d}E{int(episode):02d}"
                    if episode_title and not episode_title.startswith('S') and episode_title != f"Epizoda {episode}":
                        display_title += f" {episode_title}"
            else:
                display_title = f"{title} ({year})" if year else title
            
            list_item = xbmcgui.ListItem(label=display_title)
            
            if poster:
                list_item.setArt({'poster': poster})
            
            # Uložit metadata pro akce
            list_item.setProperty('tmdb_id', str(tmdb_id) if tmdb_id else '')
            list_item.setProperty('media_type', media_type)
            list_item.setProperty('title', title)
            list_item.setProperty('year', str(year) if year else '')
            
            # Pro seriály uložíme i info o epizodě
            if media_type == 'series':
                list_item.setProperty('season', str(item.get('season', '')))
                list_item.setProperty('episode', str(item.get('episode', '')))
                list_item.setProperty('episode_title', item.get('episode_title', ''))
            
            list_control.addItem(list_item)
        
        # Pokud máme položky, nastavit preview pro první
        if list_control.size() > 0:
            self._update_preview_for_current_item()
    
    def _populate_preview_actions(self, media_type, tmdb_id, title, year):
        """Přetíženo - upravené akční tlačítka pro Continue Watching."""
        try:
            list_control = self.getControl(6100)
            list_control.reset()
            
            # Získat aktuální index pro smazání
            poster_list = self.getControl(2000)
            current_index = poster_list.getSelectedPosition()
            
            if media_type == 'movie':
                actions = [
                    {'label': 'Přehrát', 'action': 'play'},
                    {'label': 'Detaily', 'action': 'details'},
                    {'label': 'Moje filmy', 'action': 'add_list'},
                    {'label': 'Knihovna', 'action': 'add_library'},
                    {'label': 'Smazat', 'action': 'remove_from_history'},
                ]
            elif media_type in ['series', 'tv']:
                # Pro seriály: Přehrát (aktuální epizodu), Další díl, atd.
                actions = [
                    {'label': 'Přehrát', 'action': 'play_episode'},
                    {'label': 'Další díl', 'action': 'next_episode'},
                    {'label': 'Sezóny', 'action': 'seasons'},
                    {'label': 'Moje seriály', 'action': 'add_list'},
                    {'label': 'Smazat', 'action': 'remove_from_history'},
                ]
            else:
                actions = [
                    {'label': 'Smazat', 'action': 'remove_from_history'},
                ]
            
            for act in actions:
                item = xbmcgui.ListItem(label=act['label'])
                item.setProperty('action', act['action'])
                item.setProperty('tmdb_id', str(tmdb_id))
                item.setProperty('title', title)
                item.setProperty('year', year)
                item.setProperty('media_type', media_type)
                item.setProperty('history_index', str(current_index))
                list_control.addItem(item)
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] ContinueWatching preview actions error: {e}", xbmc.LOGERROR)
    
    def _handle_preview_action_click(self):
        """Přetíženo - zpracování akcí specifických pro Continue Watching."""
        try:
            list_control = self.getControl(6100)
            item = list_control.getSelectedItem()
            if not item:
                return
            
            action = item.getProperty('action')
            tmdb_id = item.getProperty('tmdb_id')
            title = item.getProperty('title')
            year = item.getProperty('year')
            media_type = item.getProperty('media_type')
            history_index = item.getProperty('history_index')
            
            if action == 'next_episode':
                self._handle_next_episode(title, tmdb_id, history_index)
            elif action == 'play_episode':
                self._handle_play_episode(history_index)
            elif action == 'remove_from_history':
                self._handle_remove_from_history(history_index)
            elif action == 'play':
                # Přehrát film pomocí klasického search dialogu
                self._handle_play_movie(history_index)
            else:
                # Delegovat na parent pro standardní akce (details, add_list, atd.)
                super(ContinueWatchingDialog, self)._handle_preview_action_click()
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] ContinueWatching action error: {e}", xbmc.LOGERROR)
    
    def _handle_next_episode(self, series_name, tmdb_id, history_index):
        """Přehraje další díl seriálu."""
        try:
            idx = int(history_index) if history_index else 0
            if idx < 0 or idx >= len(self._history_items):
                return
            
            history_item = self._history_items[idx]
            current_season = history_item.get('season')
            current_episode = history_item.get('episode')
            
            if current_season is None or current_episode is None:
                xbmcgui.Dialog().notification(
                    self.addon.getAddonInfo('name'),
                    "Chybí informace o aktuální epizodě",
                    xbmcgui.NOTIFICATION_WARNING
                )
                return
            
            # Použít existující funkci
            from wcs.utils import play_next_episode_with_metadata
            
            params = {
                'tmdb_id': tmdb_id,
                'current_season': current_season,
                'current_episode': current_episode,
                'series_name': series_name
            }
            
            # Zavřít dialog a přehrát
            self.close()
            play_next_episode_with_metadata(params, self.addon)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Next episode error: {e}", xbmc.LOGERROR)
    
    def _handle_play_movie(self, history_index):
        """Přehraje film z historie pomocí klasického search dialogu."""
        try:
            idx = int(history_index) if history_index else 0
            if idx < 0 or idx >= len(self._history_items):
                return
            
            history_item = self._history_items[idx]
            
            from wcs.utils import play_movie_from_addon
            
            # Sestavit meta slovník z historie
            meta = {
                'title': history_item.get('title', ''),
                'year': history_item.get('year', ''),
                'plot': history_item.get('plot', ''),
                'poster': history_item.get('poster', ''),
                'fanart': '',
                'rating': history_item.get('rating', ''),
                'genre': history_item.get('genre', ''),
                'director': '',
                'runtime': '',
                'tmdb_id': history_item.get('tmdb_id'),
            }
            
            # Zavřít dialog před přehráváním
            self.close()
            
            # Spustit klasické přehrávání (používá standardní search dialog)
            play_movie_from_addon(meta, self.addon)
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Play movie error: {e}", xbmc.LOGERROR)
    
    def _handle_play_episode(self, history_index):
        """Přehraje aktuální epizodu z historie."""
        try:
            idx = int(history_index) if history_index else 0
            if idx < 0 or idx >= len(self._history_items):
                return
            
            history_item = self._history_items[idx]
            
            from wcs.utils import search_and_play_episode
            
            self.close()
            search_and_play_episode(
                series_name=history_item.get('title', ''),
                season=history_item.get('season', 1),
                episode=history_item.get('episode', 1),
                ep_name=history_item.get('episode_title', ''),
                plot=history_item.get('plot', ''),
                poster=history_item.get('poster', ''),
                fanart='',
                rating=history_item.get('rating', ''),
                genre=history_item.get('genre', ''),
                addon=self.addon,
                year=history_item.get('year', ''),
                runtime='',
                tmdb_id=history_item.get('tmdb_id', '')
            )
            
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Play episode error: {e}", xbmc.LOGERROR)
    
    def _handle_remove_from_history(self, history_index):
        """Smaže položku z historie."""
        try:
            idx = int(history_index) if history_index else 0
            if idx < 0 or idx >= len(self._history_items):
                return
            
            removed_item = self._history_items[idx]
            item_title = removed_item.get('title', 'Neznámý')
            
            # Potvrzovací dialog
            dialog = xbmcgui.Dialog()
            if not dialog.yesno("Smazat z historie", f"Opravdu chcete smazat '{item_title}' z historie?"):
                return
            
            # Smazat položku
            self._history_items.pop(idx)
            user_data.store_recently_played(self.addon, self._history_items)
            
            # Notifikace
            xbmcgui.Dialog().notification(
                self.addon.getAddonInfo('name'),
                f"'{item_title}' smazáno z historie",
                xbmcgui.NOTIFICATION_INFO
            )
            
            # Obnovit poster flow
            if self._history_items:
                self._populate_from_history()
            else:
                # Historie je prázdná - zavřít dialog
                self.close()
                
        except Exception as e:
            xbmc.log(f"[plugin.video.milionar] Remove from history error: {e}", xbmc.LOGERROR)
    
    def onClick(self, controlId):
        """Přetíženo pro handling speciálních tlačítek."""
        # AI tlačítka
        if controlId == 4000:
            list_control = self.getControl(4000)
            item = list_control.getSelectedItem()
            if not item:
                return
            
            val = item.getProperty('value')
            label = item.getLabel()
            
            if val == 'recommend_from_history':
                # Spustit AI doporučení s kontextem historie (mix filmů a seriálů)
                self._start_ai_from_history('mixed')
            elif val == 'recommend_similar_movies':
                self._start_ai_from_history('movie')
            elif val == 'recommend_similar_series':
                self._start_ai_from_history('series')
            else:
                # Delegovat na parent
                super(ContinueWatchingDialog, self).onClick(controlId)
        else:
            # Delegovat na parent pro ostatní controls
            super(ContinueWatchingDialog, self).onClick(controlId)
    
    def _start_ai_from_history(self, media_filter='mixed'):
        """
        Spustí AI doporučení na základě historie.
        
        Args:
            media_filter: 'movie', 'series', nebo 'mixed'
        """
        # Filtrovat položky podle typu
        if media_filter == 'movie':
            filtered = [i for i in self._history_items if i.get('media_type') == 'movie']
            self.media_type = 'movie'
        elif media_filter == 'series':
            filtered = [i for i in self._history_items if i.get('media_type') == 'series']
            self.media_type = 'series'
        else:
            filtered = self._history_items
            self.media_type = 'movie'  # Default, ale kontext bude mixed
        
        if not filtered:
            self.append_system_message(f"V historii nejsou žádné {'filmy' if media_filter == 'movie' else 'seriály' if media_filter == 'series' else 'položky'}.")
            return
        
        # Nastavit kontext
        self.context = {
            'type': 'recently_played',
            'media_type': self.media_type if media_filter != 'mixed' else 'mixed',
            'data': filtered
        }
        
        # Inicializovat prompt builder
        from wcs.ai.DialogAIChatRecommendation import AIRecommendationPromptBuilder
        self._prompt_builder = AIRecommendationPromptBuilder(self.media_type, self.context)
        
        # Vytvořit zprávu pro uživatele
        if media_filter == 'movie':
            user_msg = "Doporuč mi filmy na základě mé historie sledování."
        elif media_filter == 'series':
            user_msg = "Doporuč mi seriály na základě mé historie sledování."
        else:
            user_msg = "Doporuč mi filmy a seriály na základě mé historie sledování."
        
        # Přidat do historie a spustit
        self.history.append({"role": "user", "content": user_msg})
        self.append_chat_message("Ty", user_msg)
        
        # Spustit loading animaci
        self._start_button_loading("Analyzuji historii...")
        
        # Zavolat AI
        self._start_ai_conversation(is_initial_context=True)


def show_continue_watching_dialog(addon):
    """Zobrazí dialog Pokračovat ve sledování."""
    # Nejdřív zkontrolovat, jestli je historie neprázdná
    items = user_data.load_recently_played(addon)
    if not items:
        xbmcgui.Dialog().notification(
            addon.getAddonInfo('name'),
            "Historie je prázdná",
            xbmcgui.NOTIFICATION_INFO
        )
        return
    
    addon_path = addon.getAddonInfo('path')
    
    # Použijeme SDÍLENÝ XML z AI Chat dialogu
    dialog = ContinueWatchingDialog(
        "ai/DialogAIChatRecommendation.xml",
        addon_path,
        "Default",
        "1080i"
    )
    dialog.doModal()
    del dialog
